% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/3_fit_functions.R
\name{BCSreg}
\alias{BCSreg}
\alias{print.BCSreg}
\title{Box-Cox Symmetric Regression for Positive Data}
\usage{
BCSreg(
  formula,
  data,
  subset,
  na.action,
  family = "NO",
  zeta,
  link = "log",
  sigma.link = "log",
  alpha.link,
  control = BCSreg.control(...),
  model = FALSE,
  y = FALSE,
  x = FALSE,
  ...
)

\method{print}{BCSreg}(x, digits = max(3, getOption("digits") - 3), ...)
}
\arguments{
\item{formula}{a symbolic description of the model, allowing the specification of
different regression structures for model parameters using the \code{\link[Formula]{Formula}} package.
By default, the formula defines the regression structure for the scale parameter.
It can include up to three parts, separated by the `|` operator:

\itemize{
  \item \bold{First part:} specifies the model for the scale parameter.
  \item \bold{Second part (optional):} defines a regression structure for the relative dispersion parameter.
  \item \bold{Third part (only applicable for zero-adjusted positive data):} models the zero-adjustment parameter.
}

See "Details" for further explanation.}

\item{data, subset, na.action}{arguments controlling formula processing via
\code{\link[stats]{model.frame}}.}

\item{family}{a character that specifies the symmetric generating family of the BCS distribution.
Available options are: \code{"NO"} (default), \code{"ST"}, \code{"LOI"}, \code{"LOII"},
\code{"PE"}, \code{"SN"}, \code{"HP"}, and \code{"SL"}, corresponding to the normal,
Student-\emph{t}, type I logistic, type II logistic, power exponential, sinh-normal,
hyperbolic, and slash distributions, respectively.}

\item{zeta}{Strictly positive extra parameter. It must be specified with a single value in cases
where the BCS distribution has an extra parameter.}

\item{link, sigma.link}{character specification of the link functions for
the scale and relative dispersion regression structures, respectively.
Currently, \code{"log"} (default), \code{"sqrt"}, \code{"1/mu^2"},
\code{"inverse"}, and \code{"identity"} are supported.}

\item{alpha.link}{character specification of the link function for the
zero-adjustment regression structure (only applicable for zero-inflated positive data). Currently, \code{"logit"} (default),
\code{"probit"}, \code{"cloglog"}, \code{"cauchit"}, and \code{"identity"}
are supported.}

\item{control}{a list of control parameters passed as arguments for
the \code{\link[stats]{optim}} function specified via \code{\link{BCSreg.control}}.}

\item{model, y, x}{logicals. If \code{TRUE}, the corresponding components of the fit
(the model frame, the response, and the model matrices, respectively) are returned. For
\code{print()}, \code{x} is a fitted model object of class \code{"BCSreg"}.}

\item{...}{arguments passed to \code{\link{BCSreg.control}}. A particularly relevant argument is
\code{lambda = 0}, which specifies the correspondent log-symmetric or zero-adjusted
regression model.}

\item{digits}{a non-null value for digits specifies the minimum number of significant digits to
be printed in values.}
}
\value{
The \code{BCSreg} function returns an object of class \code{"BCSreg"},
     which consists of a list with the following components:
 \describe{
    \item{coefficients}{a list containing the elements \code{"mu"} and
        \code{"sigma"} that consist of the estimates of the coefficients
         associated with the scale and relative dispersion regression structures,
         respectively. If the model is zero-adjusted, the element \code{"alpha"}
         will also be returned with the estimated coefficients for the regression
         structure of the zero-adjustment parameter.}
    \item{fitted.values}{a vector with the fitted median responses. Not to be
         confused with the fitted values for \code{mu}.}
    \item{mu}{a vector with the fitted scale parameters (or conditional scale parameters for a zero-adjusted fit).}
    \item{sigma}{a vector with the fitted relative dispersion parameters (or conditional relative dispersion parameters for a zero-adjusted fit).}
    \item{lambda}{ the maximum likelihood estimate of the skewness parameter (\code{lambda}), or its fixed value
        specified in the \code{BCSreg} function.}
    \item{zeta}{the specified value for the extra parameter of the corresponding
        BCS distribution, if applicable.}
    \item{family}{the generating family of the fitted BCS distribution.}
    \item{link}{a list with elements \code{"mu"} and \code{"sigma"} with the
        specified link functions for the \code{mu} and \code{sigma} regression
        structures, respectively. If the model is zero-adjusted, the element
        \code{"alpha"} will also be returned with the link function for
        the regression structure of the zero-adjustment parameter.}
    \item{logLik}{log-likelihood of the fitted model.}
    \item{vcov}{asymptotic covariance matrix of the estimators. By default, the asymptotic
        covariance matrix is based on a analytical expression of the observed information matrix.
        It can be obtained numerically based on the Hessian matrix via \code{\link[stats]{optim}}
        if the argument \code{hessian = TRUE} is used in the \code{BCSreg} function.}
    \item{nobs}{number of observations.}
    \item{df.null}{residual degrees of freedom in the null model (a model without
        any regression structure).}
    \item{df.residual}{residual degrees of freedom in the fitted model, that is,
        the sample size minus the number of model parameters.}
    \item{control}{the control arguments passed to the optim call.}
    \item{start}{a vector with the starting values used in the iterative process.}
    \item{optim}{a list with the output from \code{\link[stats]{optim}}.}
    \item{converged}{logical indicating successful convergence of the iterative
        process.}
    \item{call}{the original function call.}
    \item{formula}{the formula used.}
    \item{terms}{a list with elements "\code{mu}", "\code{sigma}", and "\code{full}" containing
        the term objects for the respective models. If the model is zero-adjusted,
        the element \code{"alpha"} will also be returned with the term object for the
        zero-adjustment model.}
    \item{levels}{a list with elements "\code{mu}", "\code{sigma}", and "\code{full}" containing
        the levels of the categorical regressors. If the model is zero-adjusted, the element
        \code{"alpha"} will also be returned.}
    \item{contrasts}{a list with elements "\code{mu}" and "\code{sigma}"
        containing the contrasts corresponding to levels from the respective models. If the model is zero-adjusted, the element
        \code{"alpha"} will also be returned. }
    \item{model}{the full model frame (if \code{y = TRUE}).}
    \item{y}{the response variable (if \code{y = TRUE}).}
    \item{x}{a list with elements "\code{mu}" and "\code{sigma}" with the model matrices from
        the \code{mu} and \code{sigma} submodels (if \code{x = TRUE}). If the model is zero-adjusted, the element
        \code{"alpha"} will also be returned with the model matrix for the
        zero-adjustment submodel.}
    \item{alpha}{a vector with the fitted zero-adjustment parameters when a zero-adjusted
        model is considered; and \code{NULL}, otherwise.}
   }
}
\description{
Fit a Box-Cox symmetric (BCS) or a zero-adjusted BCS regression model
    using maximum likelihood estimation.
}
\details{
The \code{BCSreg} function implements maximum likelihood estimation in the
    class of the BCS and zero-adjusted BCS regression models for the analysis of positive
    or zero-inflated data (Medeiros and Queiroz, 2025). The BCS distributions
    (Ferrari and Fumes, 2017) are a broad class of flexible distributions that achieve different
    levels of skewness and tail-heaviness. See details in \link{BCS}.

    The distributions currently implemented in the \code{BCSreg} package, along
    with their abbreviations used in the \code{family} argument, are listed below:
    \tabular{llc}{
    \bold{Distribution}  \tab \bold{Family abbreviation} \tab \bold{N. of extra parameters}\cr
    Box-Cox Hyperbolic  \tab \code{"HP"}      \tab  1  \cr
    Box-Cox Type I Logistic  \tab \code{"LOI"}      \tab  0  \cr
    Box-Cox Type II Logistic  \tab \code{"LOII"}      \tab  0  \cr
    Box-Cox Normal  \tab \code{"NO"}      \tab  0  \cr
    Box-Cox Power Exponential  \tab \code{"PE"}      \tab  1  \cr
    Box-Cox Sinh-Normal  \tab \code{"SN"}      \tab  1  \cr
    Box-Cox Slash  \tab \code{"SL"}      \tab  1  \cr
    Box-Cox \emph{t}  \tab \code{"ST"}      \tab  1  \cr
    }

    The BCS distributions have at least three parameters: scale (\code{mu}),
    relative dispersion (\code{sigma}), and skewness (\code{lambda}) parameters.
    Some distributions may also depend on an additional parameter (\code{zeta}),
    such as the Box-Cox \emph{t} and Box-Cox power exponential distributions. The
    BCS distributions reduce to the log-symmetric distributions
    (Vanegas and Paula, 2016) when \code{lambda} is fixed at zero. The log-symmetric
    distributions are an important class of probability models for positive data,
    which includes well-known distributions such as the log-normal and log-\emph{t}
    distributions. The \code{BCSreg} function allows fitting a log-symmetric
    regression using \code{lambda = 0} as an argument
    (see \code{\link{BCSreg.control}}).

    The \code{BCSreg} function also implements the class of zero-adjusted BCS regression models,
    for analyzing mixed data that is positive but can assume zero. In this case, the
    zero-adjusted BCS model corresponding to the \code{family} is fitted to the data. In
    addition to the parameters \code{mu}, \code{sigma}, \code{lambda}, and possibly \code{zeta},
    the zero-adjusted BCS distributions have a zero-adjustment parameter (\code{alpha}) that
    corresponds to the probability of observing a zero response. This parameter can also be
    modeled with a regression structure.

    The \code{formula} argument defines the regression structures for different model
    parameters using the \code{\link[Formula]{Formula}} package (Zeileis and Croissant, 2010).
    It can have up to three parts, separated by the "\code{|}" operator:
    \itemize{
     \item \bold{First part:} specifies the model for the scale parameter.
     \item \bold{Second part (optional):} defines a regression structure for the relative dispersion parameter.
     \item \bold{Third part (only applicable for zero-inflated positive data):} models the zero-adjustment parameter.
    }

    If only the first part is provided, the model applies only to the scale parameter. When a
    second part is included, a regression structure is defined for the relative dispersion
    parameter. If the data contain zero inflation, a third part can be specified to model the
    zero-adjustment parameter. If the third part is provided but the response variable contains
    no zeros, this part is ignored.

    For instance, consider a dataset where \code{y} is the zero-inflated
    dependent variable, and \code{x}, \code{s}, and \code{z} are explanatory
    variables associated with the scale, relative dispersion, and zero-adjustment
    parameters, respectively. The following formulas illustrate different model structures:
    \itemize{
     \item \code{y ~ x}         # Scale parameter only
     \item \code{y ~ x | s}     # Scale and relative dispersion parameters
     \item \code{y ~ x | s | z} # Scale, relative dispersion, and zero-adjustment parameters
     \item \code{y ~ x | 1 | z} # Scale and zero-adjustment parameters
     \item \code{y ~ 1 | s | z} # Relative dispersion and zero-adjustment parameters
    }
}
\examples{
# BCS regression for strictly positive response variables

## Data set: raycatch (for description, run ?raycatch)
hist(raycatch$cpue, xlab = "Catch per unit effort")
plot(cpue ~ tide_phase, raycatch, pch = 16,
     xlab = "Tide phase", ylab = "Catch per unit effort")
plot(cpue ~ location, raycatch, pch = 16,
     xlab = "Location", ylab = "Catch per unit effort")
plot(cpue ~ max_temp, raycatch, pch = 16,
     xlab = "Maximum temperature", ylab = "Catch per unit effort")

## Fit examples

### Fit a single Box-Cox normal regression model:
fit_bcno1 <- BCSreg(cpue ~ location + tide_phase + max_temp, raycatch)
fit_bcno1

### Fit a double Box-Cox normal regression model:
fit_bcno2 <- BCSreg(cpue ~ location + tide_phase |
                      location + tide_phase + max_temp, raycatch)
fit_bcno2


### Fit a double Box-Cox power exponential regression model (family = "PE"):
fit_bcpe <- BCSreg(cpue ~ location + tide_phase + max_temp |
                     location + tide_phase + max_temp, raycatch, family = "PE", zeta = 4)
fit_bcpe

### Fit a double log-power exponential regression model (lambda = 0):
fit_lpe <- BCSreg(cpue ~ location + tide_phase + max_temp |
                    location + tide_phase + max_temp, raycatch, family = "PE",
                  zeta = 4, lambda = 0)
fit_lpe

# Zero-adjusted BCS (ZABCS) regression for nonnegative response variables

## Data set: renewables2015 (for description, run ?renewables2015)
plot(ecdf(renewables2015$renew_elec_output), cex = 0.3, main = "Empirical CDF")
abline(h = mean(renewables2015$renew_elec_output == 0), col = "grey", lty = 3)
text(1250, 0.155, paste0("prop. of zeros: ~0.12"), col = "blue")

plot(renew_elec_output ~ adj_sav_edu, renewables2015, pch = 16,
     xlab = "Education expenditure (percent of GNI)",
     ylab = "Renewable electricity output (in TWh)")
plot(renew_elec_output ~ agri_land, renewables2015, pch = 16,
     xlab = "Matural logarithm of total agricultural land area",
     ylab = "Renewable electricity output (in TWh)")

## Fit examples

### Fit a single zero-adjusted Box-Cox normal regression model:
fit_zabcno1 <- BCSreg(renew_elec_output ~ adj_sav_edu + agri_land, renewables2015)
fit_zabcno1

### Fit a double zero-adjusted Box-Cox normal regression model:
fit_zabcno2 <- BCSreg(renew_elec_output ~ adj_sav_edu + agri_land |
                        adj_sav_edu + agri_land, renewables2015)
fit_zabcno2

### Fit a triple zero-adjusted Box-Cox normal regression model:
fit_zabcno3 <- BCSreg(renew_elec_output ~ adj_sav_edu + agri_land |
                        adj_sav_edu + agri_land |
                        adj_sav_edu + agri_land, renewables2015)
fit_zabcno3


### Fit a triple zero-adjusted Box-Cox power exponential regression model (family = "PE"):
fit_zabcpe <- BCSreg(renew_elec_output ~ adj_sav_edu + agri_land |
                     adj_sav_edu + agri_land |
                     adj_sav_edu + agri_land, renewables2015, family = "PE", zeta = 4)
fit_zabcpe

### Fit a triple zero-adjusted log-power exponential regression model (lambda = 0):
fit_zalpe <- BCSreg(renew_elec_output ~ adj_sav_edu + agri_land |
                    adj_sav_edu + agri_land |
                    adj_sav_edu + agri_land, renewables2015, family = "PE",
                  zeta = 4, lambda = 0)
fit_zalpe
}
\references{
Cribari-Neto F, Zeileis A (2010). Beta Regression in R. \emph{Journal of Statistical
    Software}, \bold{34}, 1---24

    Ferrari, S. L. P., and Fumes, G. (2017). Box-Cox symmetric distributions and
    applications to nutritional data. \emph{AStA Advances in Statistical Analysis},
    \bold{101}, 321---344.

    Medeiros, R. M. R., and Queiroz, F. F. (2025). Flexible modeling of nonnegative continuous
    data: Box-Cox symmetric regression and its zero-adjusted extension.

    Vanegas, L. H., and Paula, G. A. (2016). Log-symmetric distributions:
    statistical properties and parameter estimation. \emph{Brazilian Journal of
    Probability and Statistics}, \bold{30},196---220

    Zeileis A, Croissant Y (2010). Extended Model Formulas in R: Multiple Parts and Multiple
    Responses. \emph{Journal of Statistical Software}, \bold{34}, 1---13.
}
\seealso{
\code{\link{summary.BCSreg}} for more detailed summaries,
\code{\link{residuals.BCSreg}} to extract residuals from the fitted model,
\code{\link{plot.BCSreg}} for diagnostic plots, and
\code{\link{extra.parameter}} for selection of the extra parameter \code{zeta} via Upsilon goodness-of-fit statistic and profile likelihood.
Information on additional methods for \code{"BCSreg"} objects can be found at \code{\link{BCSreg-methods}}.
}
\author{
Francisco F. de Queiroz <\email{felipeq@ime.usp.br}>

Rodrigo M. R. de Medeiros <\email{rodrigo.matheus@ufrn.br}>
}
