#' Import actigraphy with header stripping and harmonized `Activity`
#'
#' Reads a CSV/TSV, automatically strips any leading header lines, selects user
#' specified `Date` and `Time` columns, and harmonizes one chosen activity
#' column to the canonical name `Activity`. Adds an `id` (defaults to the file
#' stem if not supplied).
#'
#' @param file Character path to the actigraphy file.
#' @param date_col,time_col Character names of the date and time columns in `file`.
#' @param activity_cols Character vector of candidate activity columns; the first
#'   that exists will be used and renamed to `Activity`.
#' @param id Optional subject id (character). Defaults to the filename stem.
#' @param keep_extra Logical; if `FALSE` keeps only `id, Date, Time, Activity`,
#'   otherwise preserves extra columns present in the file.
#' @param drop_original_activity_cols Logical; if TRUE, drop the original activity
#' @return A `data.frame` with at least columns `id`, `Date`, `Time`, `Activity`.
#'
#' @details
#' Internally uses `data.table::fread()` with automatic header detection
#' (skips lines before the first row that looks like a header). No time zone
#' conversion is performed.
#'
#' @seealso [screen_wear()]
#' @export

import_acti_file <- function(file,
                             date_col,
                             time_col,
                             activity_cols,        # e.g., c("VM","vector magnitude","Counts")
							 id = NULL,
                             keep_extra   = FALSE, # keep all other columns?
                             drop_original_activity_cols = TRUE) {
  if (!requireNamespace("data.table", quietly = TRUE)) stop("Needs data.table")
  if (!requireNamespace("tibble", quietly = TRUE))     stop("Needs tibble")

  norm <- function(x) tolower(gsub("[^a-z0-9]", "", x))

  # 1) Detect header row (first line containing BOTH date_col & time_col)
  header_lines <- readLines(file, n = 100, warn = FALSE)
  if (!length(header_lines)) stop("File appears empty.")
  nd <- norm(date_col); nt <- norm(time_col)
  has_both <- function(line) {
    ln <- norm(line)
    grepl(nd, ln, fixed = TRUE) && grepl(nt, ln, fixed = TRUE)
  }
  header_row <- which(vapply(header_lines, has_both, logical(1)))[1]
  if (is.na(header_row)) stop(sprintf("No header line with '%s' and '%s' found.", date_col, time_col))

  # 2) Read table starting at header
  df <- data.table::fread(
    file,
    skip        = header_row - 1L,
    header      = TRUE,
    strip.white = TRUE,
    fill        = TRUE,
    data.table  = FALSE
  )

  # 3) Map names (case/format-insensitive)
  nm  <- names(df)
  nmn <- norm(nm)

  pick_one <- function(target) {
    i <- match(norm(target), nmn)
    if (is.na(i)) stop(sprintf("Column '%s' not found.", target))
    nm[i]
  }
  pick_first_present <- function(cands) {
    # return the first candidate that exists (after normalization)
    for (cand in cands) {
      i <- match(norm(cand), nmn)
      if (!is.na(i)) return(nm[i])
    }
    stop(sprintf("None of the activity columns found: %s", paste(activity_cols, collapse = ", ")))
  }

  dcol <- pick_one(date_col)
  tcol <- pick_one(time_col)
  acol <- pick_first_present(activity_cols)

  # 4) Add id
  id <- tools::file_path_sans_ext(basename(file))
  df$id <- id

  # 5) Harmonize activity -> 'Activity'
  # If 'Activity' already exists, overwrite it with selected acol
  df$Activity <- df[[acol]]

  # 6) Build return set
  if (keep_extra) {
    # Optionally drop the original activity columns if requested
    if (drop_original_activity_cols) {
      drop_these <- setdiff(unique(c(acol)), "Activity")
      df <- df[, setdiff(names(df), drop_these), drop = FALSE]
    }
    return(tibble::as_tibble(df))
  } else {
    keep <- unique(c("id", dcol, tcol, "Activity"))
    return(tibble::as_tibble(df[, keep, drop = FALSE]))
  }
}
