% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DRclass.R
\name{DRclass_k_Cdf}
\alias{DRclass_k_Cdf}
\title{Calculate lower and upper bounds of the cumulative density function for the
marginals of a Density Ratio Class for which the lower and upper bounding
functions are proportional.}
\usage{
DRclass_k_Cdf(sample_u, k = 1, nout = 512)
}
\arguments{
\item{sample_u}{Sample from a distribution proportional to the upper bound
of the class, often from the posterior of the upper bound
of the prior in Bayesian inference.
Columns represent variables, rows go across the sample.}

\item{k}{Factor of proportionality between upper (u) and lower (l)
bound: u = k * l}

\item{nout}{Number of equally spaced output intervals for the marginal
densities.}
}
\value{
Three dimensional array with the following dimensions:
                1: variable corresponding to column of the sample
                2: equidistant spacing of that variable
                3: five columns for
                   variable values,
                   lower bound of the cdf of the marginal of the class,
                   cdf of the marginal corresponding to the lower class boundary,
                   cdf of the marginal corresponding to the upper class boundary,
                   upper bound of the cdf of the marginal of the class
}
\description{
The cumulative density functions for the marginals of a distributions
proportional to the lower and upper bounding functions are also provided.
}
\details{
This function is more efficient than 'DRclass_lu_Pdf' as it does not
need the evaluation of the bounding functions, l and u. 
It is thus recommended to use this function if l and u are proportional.
}
\examples{
# example of the application of DRclass functions:
# ------------------------------------------------

# parameter values
k        <- 10
sd       <- 0.5
sampsize <- 10000

# upper and lower class boundaries:
u <- function(x) { return(    dnorm(x,0,sd)) }
l <- function(x) { return(1/k*dnorm(x,0,sd)) }

# generate sample:
sample_u <- cbind(rnorm(sampsize,0,sd),rnorm(sampsize,0,sd))  # example of 2d sample

# get class boundaries (back from sample):
pdf1   <- DRclass_k_Pdf(sample_u,k=k,adjust=2)       # faster for l proportional to u
pdf2   <- DRclass_lu_Pdf(sample_u,l=l,u=u,adjust=2)  # l and u could have different shapes

# get cdf bounds:
cdf1   <- DRclass_k_Cdf(sample_u,k=k)
cdf2   <- DRclass_lu_Cdf(sample_u,l=l,u=u)

# get quantile bounds:
quant1 <- DRclass_k_Quantile(sample_u,k=k,probs=c(0.025,0.5,0.975))
quant2 <- DRclass_lu_Quantile(sample_u,l=l,u=u,probs=c(0.025,0.5,0.975))

# plot selected features of the first component of the sample:
oldpar <- par(no.readonly=TRUE)
par(mar=c(5, 4, 1, 4) + 0.1)  # c(bottom, left, top, right)
plot(pdf1[1,,c("x","u")],type="l",xaxs="i",yaxs="i",xlim=c(-2,2),xlab="x",ylab="pdf")
lines(pdf2[1,,c("x","l")])
par(new=TRUE)
plot(cdf1[1,,c("x","F_upper")],type="l",xaxs="i",yaxs="i",axes=FALSE,
     xlim=c(-2,2),ylim=c(0,1),ylab="",lty="dashed")
axis(4); mtext("cdf",4,2)
lines(cdf2[1,,c("x","F_lower")],lty="dashed")
abline(v=quant1["quant_lower_0.5",1],lty="dotted")
abline(v=quant1["quant_upper_0.5",1],lty="dotted")
abline(v=quant1["quant_lower_0.025",1],lty="dotdash")
abline(v=quant1["quant_upper_0.975",1],lty="dotdash")
par(oldpar)
}
