% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_by_class.R
\name{plot_by_class}
\alias{plot_by_class}
\title{Summarise and plot standing volume by c130 class and species}
\usage{
plot_by_class(
  data,
  volume_col = "dagnelie_vc22_2",
  breaks = seq(30, 230, by = 25),
  small_limit = 60,
  medium_limit = 120,
  output = NULL,
  make_plot = TRUE
)
}
\arguments{
\item{data}{A data frame containing at least:
\itemize{
\item \code{c130}: stem circumference at 1.30 m (cm),
\item \code{species_code}: species identifier,
\item a volume column (defaults to \code{"dagnelie_vc22_2"}).
}}

\item{volume_col}{Name of the column containing tree volume (string).
Defaults to \code{"dagnelie_vc22_2"}.}

\item{breaks}{Numeric vector defining c130 class boundaries (cm).
Default is \code{seq(30, 230, by = 25)}.}

\item{small_limit}{Threshold between small and medium wood (cm of c130).
A vertical dashed line is drawn at this value in the plot. Default is 60.}

\item{medium_limit}{Threshold between medium and large wood (cm of c130).
A vertical dashed line is drawn at this value in the plot. Default is 120.}

\item{output}{Optional file path where the cross-tabulated table should be
exported as a CSV. If \code{NULL} (default), no file is written.
Export is handled by the utility function \code{export_output()}.}

\item{make_plot}{Logical; if \code{TRUE} (default), a ggplot object is
created and returned alongside the table.}
}
\value{
A list with two components:
\itemize{
\item \code{table}: data frame with species as rows and c130 classes
as columns, plus a TOTAL row.
\item \code{plot}: a \code{ggplot2} object (or \code{NULL} if
\code{make_plot = FALSE}).
}
}
\description{
This function builds a cross-tabulated volume table by species and c130
classes, adds a TOTAL row per class, optionally exports the table as a CSV,
and returns a ggplot object showing the volume distribution by c130 class.
}
\details{
The table has:
\itemize{
\item rows = species (plus a \code{"TOTAL"} row),
\item columns = c130 classes (e.g. \code{[30,55)}, \code{[55,80)}, ...),
\item cells = summed volume per species and c130 class.
}

The plot shows a volume-weighted histogram (or barchart) by c130 class,
stacked by species, with a trend line for total volume per class and
dashed vertical lines marking small, medium and large wood thresholds.

The c130 classes are built with \code{cut()} using \code{breaks} as class
boundaries and an open-ended last class (using \code{Inf} as the upper
bound). The resulting factor labels (e.g. \code{"[30,55)"}) are used as
column names in the cross-tabulated table.

For the plot, volume is used as a weight so that bar heights represent
total volume per c130 class. A trend line is computed from total volume
per class midpoint using the same binning scheme.
}
\examples{
set.seed(123)
n <- 150
c130 <- runif(n, 30, 230)
htot <- 0.25 * c130 + rnorm(n, 0, 3)
htot <- pmax(5, pmin(htot, 45))

species_list <- c(
  "PINUS_SYLVESTRIS", "PICEA_ABIES",
  "QUERCUS_ROBUR", "FAGUS_SYLVATICA", "BETULA_SP"
)
species_code <- sample(species_list, n, replace = TRUE)

df <- data.frame(
  c130 = round(c130, 1),
  htot = round(htot, 1),
  species_code = species_code
)

df <- dagnelie_vc22_2(df)
res <- plot_by_class(df, volume_col = "dagnelie_vc22_2")

res$table

print(res$plot)

}
