params.n2w <- function(J, obsdist, obspar, Pi) {

  # Transform transition probabilities to log-ratios (excluding diagonal)
  wPi <- log(Pi / diag(Pi))
  wPi <- as.vector(wPi[!diag(J)])

  # Transform observation parameters based on distribution type
  if (obsdist == "norm") {
    # Normal distribution: mean (unconstrained), log(sd)
    if (is.null(obspar$mean) || is.null(obspar$sd)) {
      stop("Missing parameters 'mean' and/or 'sd' for normal distribution.")
    }
    wmean <- obspar$mean
    wsd <- log(obspar$sd)
    wparams <- c(wmean, wsd, wPi)

  } else if (obsdist == "pois") {
    # Poisson distribution: log(lambda)
    if (is.null(obspar$lambda)) {
      stop("Missing parameter 'lambda' for Poisson distribution.")
    }
    wlambda <- log(obspar$lambda)
    wparams <- c(wlambda, wPi)

  } else if (obsdist == "weibull") {
    # Weibull distribution: log(shape), log(scale)
    if (is.null(obspar$shape) || is.null(obspar$scale)) {
      stop("Missing parameters 'shape' and/or 'scale' for Weibull distribution.")
    }
    wshape <- log(obspar$shape)
    wscale <- log(obspar$scale)
    wparams <- c(wshape, wscale, wPi)

  } else if (obsdist == "zip") {
    # Zero-inflated Poisson: log(lambda), logit(pi)
    if (is.null(obspar$lambda) || is.null(obspar$pi)) {
      stop("Missing parameters 'pi' and/or 'lambda' for ZIP distribution.")
    }
    wlambda <- log(obspar$lambda)
    wpi <- log(obspar$pi / (1 - obspar$pi))  # logit transformation
    wparams <- c(wlambda, wpi, wPi)

  } else if (obsdist == "nbinom") {
    # Negative binomial distribution: log(size), log(mu)
    if (is.null(obspar$size) || is.null(obspar$mu)) {
      stop("Missing parameters 'size' and/or 'mu' for negative binomial distribution.")
    }
    wsize <- log(obspar$size)
    wmu <- log(obspar$mu)
    wparams <- c(wsize, wmu, wPi)

  } else if (obsdist == "zinb") {
    # Zero-inflated negative binomial: log(size), log(mu), logit(pi)
    if (is.null(obspar$size) || is.null(obspar$mu) || is.null(obspar$pi)) {
      stop("Missing parameters 'size' and/or 'mu' and/or 'pi' for the zero-inflated negative binomial distribution")
    }
    wsize <- log(obspar$size)
    wmu <- log(obspar$mu)
    wpi <- log(obspar$pi / (1 - obspar$pi))  # logit transformation
    wparams <- c(wsize, wmu, wpi, wPi)

  } else if (obsdist == "exp") {
    # Exponential distribution: log(rate)
    if (is.null(obspar$rate)) {
      stop("Missing parameter 'rate' for the exponential distribution")
    }
    wrate <- log(obspar$rate)
    wparams <- c(wrate, wPi)

  } else if (obsdist == "gamma") {
    # Gamma distribution: log(shape), log(rate)
    wshape <- log(obspar$shape)
    wrate <- log(obspar$rate)
    wparams <- c(wshape, wrate, wPi)

  } else if (obsdist == "lnorm") {
    # Log-normal distribution: meanlog (unconstrained), log(sdlog)
    wmeanlog <- obspar$meanlog
    wsdlog <- log(obspar$sdlog)
    wparams <- c(wmeanlog, wsdlog, wPi)

  } else if (obsdist == "gev") {
    # Generalized extreme value: loc (unconstrained), log(scale), shape (unconstrained)
    if (is.null(obspar$loc) || is.null(obspar$scale) || is.null(obspar$shape)) {
      stop("Missing parameters 'loc' and/or 'scale' and/or 'shape' for the generalised extreme value distribution")
    }
    wloc <- obspar$loc
    wscale <- log(obspar$scale)
    wshape <- obspar$shape
    wparams <- c(wloc, wscale, wshape, wPi)

  } else if(obsdist == "ZInormal"){
    # Zero-inflated normal distribution: mean (unconstrained), log(sd), logit(pi)
    if(is.null(obspar$mean)||is.null(obspar$sd)||is.null(obspar$pi)){
      stop("Missing parameters 'mean' and/or 'sd' and/or pi for the zero-inflated normal distribution")
    }
    wmean <- obspar$mean
    wsd <- log(obspar$sd)
    wpi <- log(obspar$pi/(1-obspar$pi))
    wparams <- c(wmean, wsd, wpi, wPi)

  } else if(obsdist=="ZIgamma"){
    # Zero-inflated gamma distribution: log(shape), log(rate), logit(pi)
    if(is.null(obspar$shape)||is.null(obspar$rate)||is.null(obspar$pi)){
      stop("Missing parameters 'shape' and/or 'rate' and/or pi for the zero-inflated gamma distribution")
    }
    wshape <- log(obspar$shape)
    wrate <- log(obspar$rate)
    wpi <- log(obspar$pi/(1-obspar$pi))
    wparams <- c(wshape, wrate, wpi, wPi)

  } else {
    stop("Invalid distribution specified.")
  }

  return(wparams)
}
