# JapanAPIs - Access Japanese Data via Public APIs and Curated Datasets
# Version 0.1.1
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_japan_population

library(testthat)

test_that("get_japan_population() returns a tibble with expected structure", {
  skip_on_cran()

  result <- get_japan_population()

  # Structure
  expect_s3_class(result, "tbl_df")
  expect_equal(ncol(result), 5)

  # Column names
  expect_named(result, c("indicator", "country", "year", "value", "value_label"))

  # Column types
  expect_type(result$indicator, "character")
  expect_type(result$country, "character")
  expect_type(result$year, "integer")
  expect_type(result$value, "integer")  # World Bank returns integers
  expect_type(result$value_label, "character")
})

test_that("get_japan_population() returns correct dimensions and years", {
  skip_on_cran()

  result <- get_japan_population()

  # Number of rows (2010–2022 inclusive)
  expect_equal(nrow(result), 13)

  # Exact years expected
  expect_equal(sort(result$year), 2010:2022)

  # Should be sorted in descending order
  expect_true(all(diff(result$year) <= 0))
})

test_that("get_japan_population() returns consistent values for Japan", {
  skip_on_cran()

  result <- get_japan_population()

  # Country should always be Japan
  expect_true(all(result$country == "Japan"))

  # Indicator should always be "Population, total"
  expect_true(all(grepl("Population, total", result$indicator, ignore.case = TRUE)))

  # Values must be positive and realistic (Japan 120M–130M)
  expect_true(all(result$value > 120e6))
  expect_true(all(result$value < 130e6))
})

test_that("get_japan_population() returns valid formatted labels", {
  skip_on_cran()

  result <- get_japan_population()

  # value_label must match formatted value
  formatted <- scales::comma(result$value, accuracy = 1)
  expect_equal(result$value_label, formatted)

  # All labels should contain commas
  expect_true(all(grepl(",", result$value_label)))
})

test_that("get_japan_population() contains non-missing values", {
  skip_on_cran()

  result <- get_japan_population()

  expect_true(all(!is.na(result$indicator)))
  expect_true(all(!is.na(result$country)))
  expect_true(all(!is.na(result$year)))
  expect_true(all(!is.na(result$value)))
  expect_true(all(!is.na(result$value_label)))
})

test_that("get_japan_population() handles API errors gracefully", {
  skip_on_cran()

  result <- tryCatch(get_japan_population(), error = function(e) NULL)

  # Should return either NULL or a tibble
  expect_true(is.null(result) || inherits(result, "tbl_df"))
})
