% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LCA.R
\name{LCA}
\alias{LCA}
\title{Fit Latent Class Analysis Models}
\usage{
LCA(
  response,
  L = 2,
  par.ini = "random",
  method = "EM",
  is.sort = TRUE,
  nrep = 20,
  starts = 100,
  maxiter.wa = 20,
  vis = TRUE,
  control.EM = NULL,
  control.Mplus = NULL,
  control.NNE = NULL
)
}
\arguments{
\item{response}{A numeric matrix of dimension \eqn{N \times I}, where \eqn{N} is the number of individuals/participants/observations
and \eqn{I} is the number of observed categorical items/variables. Each column must contain nominal-scale
discrete responses (e.g., integers representing categories).}

\item{L}{Integer specifying the number of latent classes (default: 2).}

\item{par.ini}{Specification for parameter initialization. Options include:
\itemize{
\item \code{"random"}: Completely random initialization (default).
\item \code{"kmeans"}: Initializes parameters via K-means clustering on observed data (McLachlan & Peel, 2004).
\item A \code{list} containing:
\describe{
\item{\code{par}}{An \eqn{L \times I \times K_{\max}} array of initial conditional probabilities for
each latent class, item, and response category (where \eqn{K_{\max}} is the maximum
number of categories across items).}
\item{\code{P.Z}}{A numeric vector of length \eqn{L} specifying initial prior probabilities for latent classes.}
}
}}

\item{method}{Character string specifying estimation algorithm:
\itemize{
\item \code{"EM"}: Expectation-Maximization algorithm (default).
\item \code{"NNE"}: Neural Network Estimation with transformer architecture (experimental; uses transformer + simulated
annealing, more reliable than both \code{"EM"} and \code{"Mplus"}). See \code{\link[LCPA]{install_python_dependencies}}.
\item \code{"Mplus"}: Calls external Mplus software for estimation.
Uses Mplus defaults for optimization unless overridden by \code{control.Mplus}.
}}

\item{is.sort}{A logical value. If \code{TRUE} (Default), the latent classes will be ordered in descending
order according to \code{P.Z}. All other parameters will be adjusted accordingly
based on the reordered latent classes.}

\item{nrep}{Integer controlling replication behavior:
\itemize{
\item If \code{par.ini = "random"}, number of random initializations.
\item If \code{par.ini = "kmeans"}, number of K-means runs for initialization.
\item For \code{method="Mplus"}, controls number of random starts in Mplus via \code{STARTS} option.
\item Best solution is selected by log-likelihood/BIC across replications.
\item Ignored for user-provided initial parameters.
}}

\item{starts}{Number of random initializations to explore during warm-up phase (default: 100).}

\item{maxiter.wa}{Maximum number of training iterations allowed per warm-up run. After completion,
the top \code{nrep} solutions (by log-likelihood) are promoted to final training phase (default: 20).}

\item{vis}{Logical. If \code{TRUE}, displays progress information during estimation (default: \code{TRUE}).}

\item{control.EM}{List of control parameters for EM algorithm:
\describe{
\item{\code{maxiter}}{Maximum iterations (default: 2000).}
\item{\code{tol}}{Convergence tolerance for log-likelihood difference (default: 1e-4).}
}}

\item{control.Mplus}{List of control parameters for Mplus estimation:
\describe{
\item{\code{maxiter}}{Maximum iterations for Mplus optimization (default: 2000).}
\item{\code{tol}}{Convergence tolerance for log-likelihood difference (default: 1e-4).}
\item{\code{files.path}}{A character string specifying the directory path where Mplus will write its intermediate files
(e.g., \code{.inp} model input, \code{.dat} data file, \code{.out} output, and saved posterior probabilities).
This argument is \strong{required} — if \code{NULL} (the default), the function throws an error.
The specified directory must exist and be writable; if it does not exist, the function attempts to create it recursively.
A unique timestamped subdirectory (e.g., \code{"Mplus_LCA_YYYY-MM-DD_HH-MM-SS"}) will be created within this path
to store all run-specific files and avoid naming conflicts.
If it is an empty string (\code{""}), the timestamped subdirectory \code{"Mplus_LCA_YYYY-MM-DD_HH-MM-SS"}
will be created directly under R's current working directory (\code{\link[base]{getwd}()}).}
\item{\code{files.clean}}{Logical. If \code{TRUE} (default), all intermediate files and the temporary working directory
created for this run are deleted upon successful completion or error exit (via \code{on.exit()}).
If \code{FALSE}, all generated files are retained in \code{files.path} (or the auto-generated temp dir)
for inspection or debugging. Note: when \code{files.path = NULL}, even if \code{files.clean = FALSE},
the temporary directory may still be cleaned up by the system later — for guaranteed persistence,
specify a custom \code{files.path}.}
}}

\item{control.NNE}{List of control parameters for NNE algorithm:
\describe{
\item{\code{hidden.layers}}{Integer vector specifying layer sizes in fully-connected network (default: \code{c(12,12)}).}
\item{\code{activation.function}}{Activation function (e.g., \code{"tanh"}, default: \code{"tanh"}).}
\item{\code{d.model}}{Dimensionality of transformer encoder embeddings (default: 8).}
\item{\code{nhead}}{Number of attention heads in transformer (default: 2).}
\item{\code{dim.feedforward}}{Dimensionality of transformer feedforward network (default: 16).}
\item{\code{eps}}{Small constant for numerical stability (default: 1e-8).}
\item{\code{lambda}}{A factor for slight regularization of all parameters (default: 1e-5).}
\item{\code{initial.temperature}}{Initial temperature for simulated annealing (default: 1000).}
\item{\code{cooling.rate}}{Cooling rate per iteration in simulated annealing (default: 0.5).}
\item{\code{maxiter.sa}}{Maximum iterations for simulated annealing (default: 1000).}
\item{\code{threshold.sa}}{Minimum temperature threshold for annealing (default: 1e-10).}
\item{\code{maxiter}}{Maximum training epochs (default: 1000).}
\item{\code{maxiter.early}}{Patience parameter for early stopping (default: 50).}
\item{\code{maxcycle}}{Maximum cycles for optimization (default: 10).}
\item{\code{lr}}{Learning rate, controlling the step size of neural network parameter updates (default: 0.025).}
\item{\code{scheduler.patience}}{Patience for learning rate decay (if the loss function does not improve for more than \code{patience} consecutive epochs, the learning rate will be reduced) (default: 10).}
\item{\code{scheduler.factor}}{Learning rate decay factor; the new learning rate equals the original learning rate multiplied by \code{scheduler.factor} (default: 0.70).}
\item{\code{plot.interval}}{Interval (in epochs) for plotting training diagnostics (default: 100).}
\item{\code{device}}{Specifies the hardware device; can be \code{"CPU"} (default) or \code{"GPU"}. If the GPU is not available, it automatically falls back to CPU.}
}}
}
\value{
An object of class \code{"LCA"} containing:
\describe{
\item{\code{params}}{List with estimated parameters:
\describe{
\item{\code{par}}{\eqn{L \times I \times K_{\max}} array of conditional response probabilities per latent class.}
\item{\code{P.Z}}{Vector of length \eqn{L} with latent class prior probabilities.}
}
}
\item{\code{npar}}{Number of free parameters in the model. see \code{\link[LCPA]{get.npar.LCA}}}
\item{\code{Log.Lik}}{Log-likelihood of the final model. see \code{\link[LCPA]{get.Log.Lik.LCA}}}
\item{\code{AIC}}{Akaike Information Criterion value.}
\item{\code{BIC}}{Bayesian Information Criterion value.}
\item{\code{best_BIC}}{Best BIC value across \code{nrep} runs (if applicable).}
\item{\code{P.Z.Xn}}{\eqn{N \times L} matrix of posterior class probabilities for each observation.}
\item{\code{P.Z}}{Vector of length \eqn{L} containing the prior probabilities/structural parameters/proportions for each latent class.}
\item{\code{Z}}{Vector of length \eqn{N} with MAP-classified latent class memberships.}
\item{\code{probability}}{Same as \code{params$par} (redundant storage for convenience).}
\item{\code{Log.Lik.history}}{Vector tracking log-likelihood at each EM iteration.}
\item{\code{Log.Lik.nrep}}{Vector of log-likelihoods from each replication run.}
\item{\code{model}}{The optimal neural network model object (only for \code{method="NNE"}).
Contains the trained transformer architecture corresponding to \code{best_loss}.
This object can be used for further predictions or model inspection.}
\item{\code{arguments}}{A list containing all input arguments}
}
}
\description{
This function estimates parameters of a Latent Class Analysis (LCA; Hagenaars & McCutcheon, 2002) model using
either the Expectation-Maximization (EM) algorithm or Neural Network Estimation (NNE).
It supports flexible initialization strategies and provides comprehensive model diagnostics.
}
\section{EM Algorithm}{

When \code{method = "EM"}, parameters are estimated via the Expectation-Maximization algorithm, which iterates between:
\itemize{
\item \strong{E-step:} Compute posterior class probabilities given current parameters:
\deqn{P(Z_n = l \mid \mathbf{X}_n) = \frac{\pi_l \prod_{i=1}^I P(X_{ni} = x_{ni} \mid Z_n=l)}{\sum_{k=1}^L \pi_k \prod_{i=1}^I P(X_{ni} = x_{ni} \mid Z_n=k)}}
where \eqn{x_{ni}} is the standardized (0-based) response for person \eqn{n} on item \eqn{i} (see \code{\link[LCPA]{adjust.response}}).
\item \strong{M-step:} Update parameters by maximizing expected complete-data log-likelihood:
\itemize{
\item Class probabilities: \eqn{\pi_l^{\text{new}} = \frac{1}{N} \sum_{n=1}^N P(Z_n = l \mid \mathbf{X}_n)}
\item Conditional probabilities: \eqn{P(X_i = k \mid Z=l)^{\text{new}} = \frac{\sum_{n:x_{ni}=k} P(Z_n = l \mid \mathbf{X}_n)}{\sum_{n=1}^N P(Z_n = l \mid \mathbf{X}_n)}}
}
\item \strong{Convergence}: Stops when \eqn{|\log\mathcal{L}^{(t)} - \log\mathcal{L}^{(t-1)}| < \texttt{tol}} or maximum iterations reached.
}
}

\section{Neural Network Estimation (NNE)}{

When \code{method = "NNE"}, parameters are estimated using a hybrid neural network architecture
that combines feedforward layers with transformer-based attention mechanisms. This approach jointly
optimizes profile parameters and posterior probabilities through stochastic optimization enhanced
with simulated annealing. See \code{\link[LCPA]{install_python_dependencies}}. Key components include:

\strong{Architecture}:
\describe{
\item{Input Representation}{
Observed categorical responses are converted to 0-based integer indices per item (not one-hot encoded).
For example, original responses \eqn{[1, 2, 4]} become \eqn{[0, 1, 2]}.
}
\item{Feature Estimator (Feedforward Network)}{
A fully-connected neural network with layer sizes specified by \code{hidden.layers} and activation
function \code{activation.function} processes the integer-indexed responses. This network outputs
unnormalized logits for posterior class membership (\eqn{N \times L} matrix).
}
\item{Attention Refiner (Transformer Encoder)}{
A transformer encoder with \code{nhead} attention heads that learns latent class prior probabilities
\eqn{\boldsymbol{\pi} = (\pi_1, \pi_2, \dots, \pi_L)} directly from observed responses.
\itemize{
\item Input: \code{response} matrix (\eqn{N \times I}), where \eqn{N} = observations, \eqn{I} = continuous variables.
\item Mechanism: Self-attention dynamically weighs variable importance during profile assignment, capturing complex
multivariate interactions.
\item Output: Class prior vector \eqn{\boldsymbol{\pi}} computed as the mean of posteriors:
\deqn{\pi_l = \frac{1}{N}\sum_{n=1}^N attention(\mathbf{X}_n)}
This ensures probabilistic consistency with the mixture model framework.
}
}
\item{Profile Parameter Estimation}{
Global conditional probability parameters (\eqn{P(X_i = k \mid Z = l)}) are stored as learnable
parameters \code{par} (an \eqn{L \times I \times K_{\max}} tensor). A \emph{masked softmax} is applied
along categories to enforce:
\itemize{
\item Probabilities sum to 1 within each item-class pair
\item Non-existent categories (beyond item's actual max response) are masked to zero probability
}
}
}

\strong{Optimization Strategy}:
\itemize{
\item \strong{Hybrid Training Protocol}: Alternates between:
\itemize{
\item \emph{Gradient-based phase}: AdamW optimizer minimizes negative log-likelihood with weight decay regularization:
\deqn{-\log \mathcal{L} + \lambda \|\boldsymbol{\theta}\|_2^2}
where \eqn{\lambda} is controlled by \code{lambda} (default: 1e-5). Learning rate decays adaptively when
loss plateaus (controlled by \code{scheduler.patience} and \code{scheduler.factor}).
\item \emph{Simulated annealing phase}: After gradient-based early stopping (\code{maxiter.early}), parameters
are perturbed with noise scaled by temperature:
\deqn{\theta_{\text{new}} = \theta_{\text{current}} + \mathcal{N}(0, \theta_{\text{current}} \times \frac{T}{T_0})}
Temperature \eqn{T} decays geometrically (\eqn{T \leftarrow T \times \text{cooling.rate}}) from
\code{initial.temperature} until \code{threshold.sa} is reached. This escapes poor local minima.
}
Each full cycle (gradient descent + annealing) repeats up to \code{maxcycle} times.
\item \strong{Model Selection}: Across \code{nrep} random restarts (using Dirichlet-distributed initializations
or K-means), the solution with lowest BIC is retained.
\item \strong{Diagnostics}: Training loss, annealing points, and global best solution are plotted when \code{vis=TRUE}.
}
}

\section{Mplus}{

When \code{method = "Mplus"}, estimation is delegated to external Mplus software.
The function automates the entire workflow:

\strong{Workflow}:
\describe{
\item{Temporary Directory Setup}{Creates \code{inst/Mplus} to store:
\itemize{
\item Mplus input syntax (\code{.inp})
\item Data file in Mplus format (\code{.dat})
\item Posterior probabilities output (\code{.dat})
}
Files are automatically deleted after estimation unless \code{control.Mplus$clean.files = FALSE}.
}

\item{Syntax Generation}{Constructs Mplus syntax with:
\itemize{
\item \code{CLASSES = c1(L)} specification for \eqn{L} latent classes
\item \code{CATEGORICAL} declaration for all indicator variables
\item \code{ANALYSIS} block with optimization controls:
\describe{
\item{\code{TYPE = mixture}}{Standard mixture modeling setup}
\item{\code{STARTS = starts nrep}}{Random \code{starts} and final stage optimizations}
\item{\code{STITERATIONS = maxiter.wa}}{max itertions during \code{starts}.}
\item{\code{MITERATIONS = maxiter}}{Maximum EM iterations}
\item{\code{CONVERGENCE = tol}}{Log-likelihood convergence tolerance}
}
\item \code{MODEL} block with \code{\%OVERALL\%}
}
}

\item{Execution}{Calls Mplus via \code{MplusAutomation::mplusModeler()}, which:
\itemize{
\item Converts R data to Mplus-compatible format with automatic recoding
\item Invokes Mplus executable (requires valid license and system PATH configuration)
}
}
}
}

\examples{
# Example with simulated data
set.seed(123)
data.obj <- sim.LCA(N = 500, I = 4, L = 2, IQ=0.9)
response <- data.obj$response

# Fit 2-class model with EM algorithm
\donttest{
fit.em <- LCA(response, L = 2, method = "EM", nrep = 10)
}

# Fit 2-profile model using Mplus
# need Mplus
# NOTE: 'files.path' in control.Mplus is REQUIRED — function will error if not provided.
# Example creates a timestamped subfolder (e.g., "Mplus_LCA_YYYY-MM-DD_HH-MM-SS") under './'
# to store all temporary Mplus files (.inp, .dat, .out, etc.).
\dontrun{
fit.mplus <- LCA(response, L = 2, method = "Mplus", nrep = 3,
                 control.Mplus = list(files.path = ""))
}

# Fit 2-class model with neural network estimation
# need Python
\dontrun{
fit.nne <- LCA(response, L = 2, method = "NNE", nrep = 3)
}

}
\references{
Hagenaars, J. A. , & McCutcheon, A. L. (2002). Applied Latent Class Analysis. United Kingdom: Cambridge University Press.

McLachlan, G. J., & Peel, D. (2004). Finite Mixture Models. Wiley.
https://books.google.com.sg/books?id=c2_fAox0DQoC
}
