% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mc.R
\name{mc}
\alias{mc}
\title{Monte Carlo with Tie-Breaker}
\usage{
mc(
  y,
  statistic,
  ...,
  dgp = function(y) sample(y, replace = TRUE),
  N = 99,
  type = c("geq", "leq", "absolute", "two-tailed")
)
}
\arguments{
\item{y}{A vector or data frame.}

\item{statistic}{A function or a character string that specifies
how the statistic is computed. The function needs to input the
\code{y} and output a scalar.}

\item{...}{Other named arguments for statistic which are
passed unchanged each time it is called}

\item{dgp}{A function. The function inputs the first argument
\code{y} and outputs a simulated \code{y}. It should represent the
data generating process under the null. Default value is the function
\code{sample(y, replace = TRUE)}, i.e. the
bootstrap resampling of \code{y}.}

\item{N}{An atomic vector. Number of replications of the test
statistic.}

\item{type}{A character string. It specifies the type of test
the p-value function produces. The possible values are
\code{geq}, \code{leq}, \code{absolute} and \code{two-tailed}.
 Default is \code{geq}.}
}
\value{
The returned value is an object of class \code{mc}
containing the following components:
 \item{S0}{Observed value of \code{statistic}.}
 \item{pval}{Monte Carlo p-value of \code{statistic}.}
 \item{y}{Data specified in call.}
 \item{statistic}{\code{statistic} function specified in call.}
 \item{dgp}{\code{dgp} function specified in call.}
 \item{N}{Number of replications specified in call.}
 \item{type}{\code{type} of p-value specified in call.}
 \item{call}{Original call to \code{mmc}.}
 \item{seed}{Value of \code{.Random.seed} at the start of
 \code{mc} call.}
}
\description{
Find the Monte Carlo (MC) p-value by generating N replications
of a statistic.
}
\details{
The \code{dgp} function defined by the user is used to
generate new observations in order to compute the simulated
statistics.

Then \code{\link{pvalue}} is applied to the statistic and
its simulated values. \code{\link{pvalue}} computes the
p-value by ranking the statistic compared to its simulated
values. Ties in the ranking are broken according to a uniform
distribution.

We allow for four types of p-value: \code{leq}, \code{geq},
\code{absolute} and \code{two-tailed}. For one-tailed test,
\code{leq} returns the proportion of simulated values smaller
than the statistic while \code{geq} returns the proportion of
simulated values greater than the statistic. For two-tailed
test with a symmetric statistic, one can use the
absolute value of the statistic and its simulated values to
retrieve a two-tailed test (i.e. type = \code{absolute}).
If the statistic is not symmetric, one can specify the p-value
type as \code{two-tailed} which is equivalent to twice the minimum
of \code{leq} and \code{geq}.

Ties in the ranking are broken according to a uniform
distribution.
}
\examples{
## Example 1
## Kolmogorov-Smirnov Test using Monte Carlo

# Set seed
set.seed(999)

# Generate sample data
y <- rgamma(8, shape = 2, rate = 1)

# Set data generating process function
dgp <- function(y)  rgamma(length(y), shape = 2, rate = 1)

# Set the statistic function to the Kolomogorov-Smirnov test for gamma distribution
statistic <- function(y){
    out <- ks.test(y, "pgamma", shape = 2, rate = 1)
    return(out$statistic)
}

# Apply the Monte Carlo test with tie-breaker
mc(y, statistic = statistic, dgp = dgp, N = 999, type = "two-tailed")

}
\references{
Dufour, J.-M. (2006), Monte Carlo Tests with nuisance parameters:
A general approach to finite sample inference and nonstandard asymptotics in econometrics.
\emph{Journal of Econometrics}, \bold{133(2)}, 443-447.

Dufour, J.-M. and Khalaf L. (2003), Monte Carlo Test Methods in Econometrics.
in Badi H. Baltagi, ed., \emph{A Companion to Theoretical Econometrics}, Blackwell Publishing Ltd, 494-519.
}
\seealso{
\code{\link{mmc}}, \code{\link{pvalue}}
}
