# LogisChen.R
#' Logistic-Chen Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Logistic-Chen distribution.
#'
#' The Logistic-Chen distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Logistic-Chen distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad 1-\frac{1}{1+\left[\exp \left\{\lambda\left
#' (e^{x^\beta}-1\right)\right\}-1\right]^\alpha}  \, ; \quad x \geq 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dlogis.chen()} — Density function
#'   \item \code{plogis.chen()} — Distribution function
#'   \item \code{qlogis.chen()} — Quantile function
#'   \item \code{rlogis.chen()} — Random generation
#'   \item \code{hlogis.chen()} — Hazard function
#' }
#'
#' @usage
#' dlogis.chen(x, alpha, beta, lambda, log = FALSE)
#' plogis.chen(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qlogis.chen(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rlogis.chen(n, alpha, beta, lambda)
#' hlogis.chen(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlogis.chen}: numeric vector of (log-)densities
#'   \item \code{plogis.chen}: numeric vector of probabilities
#'   \item \code{qlogis.chen}: numeric vector of quantiles
#'   \item \code{rlogis.chen}: numeric vector of random variates
#'   \item \code{hlogis.chen}: numeric vector of hazard values
#' }
#'
#' @references 
#' Joshi, R.K., & Kumar, V. (2021). 
#' Logistic Chen Distribution with Properties and Applications.
#' \emph{International Journal of Mathematics Trends and Technology (IJMTT)},
#' \bold{67(1)}, 141--151. 
#' \doi{10.14445/22315373/IJMTT-V67I1P519}   
#'  
#' @examples
#' x <- seq(0.1, 2.0, 0.1)
#' dlogis.chen(x, 1.5, 1.5, 0.1)
#' plogis.chen(x, 1.5, 1.5, 0.1)
#' qlogis.chen(0.5, 1.5, 1.5, 0.1)
#' rlogis.chen(10, 2.0, 5.0, 0.1)
#' hlogis.chen(x, 1.5, 1.5, 0.1)
#' 
#' # Data
#' x <- bladder
#' # ML estimates
#' params = list(alpha=4.46424, beta=0.15506, lambda=0.24904)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plogis.chen, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlogis.chen, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlogis.chen, pfun=plogis.chen, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LogisChen
#' @aliases dlogis.chen plogis.chen qlogis.chen rlogis.chen hlogis.chen
#' @export
NULL

#' @rdname LogisChen
#' @usage NULL
#' @export   
dlogis.chen <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	z <- exp(x[valid] ^ beta)
    u <- exp(lambda * (z-1))
    num <- alpha * beta *lambda *(x[valid] ^(beta - 1.0)) * z * u * ((u-1)^(alpha - 1.0))
    deno <- (1.0 + (u-1) ^ alpha) ^ 2.0
    pdf[valid] <- num/deno
    }
	if (log) 
       pdf<- log(pdf)
    return(pdf)   
}

#' @rdname LogisChen
#' @usage NULL
#' @export    
plogis.chen <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
    u <- exp(lambda * (exp(q[valid] ^ beta) -1))
	tmp <- 1.0 + ((u - 1.0) ^ alpha)
    cdf[valid] <- 1.0 - (1.0/tmp)
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LogisChen
#' @usage NULL
#' @export   
qlogis.chen <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	qtl[valid] <- (log((1/lambda) *(log(1 + ((p/(1 - p[valid])) ^ (1/alpha)))) + 1)) ^ (1/beta)  
    }   
    return(qtl)   
}

#' @rdname LogisChen
#' @usage NULL
#' @export  
rlogis.chen <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qlogis.chen(u, alpha, beta, lambda)
    return(q)
}

#' @rdname LogisChen
#' @usage NULL
#' @export  
hlogis.chen <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
     
    nume <- dlogis.chen(x, alpha, beta, lambda)
    surv <- 1.0 - plogis.chen(x, alpha, beta, lambda)
    hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 

