% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_track.R
\name{simulate_track}
\alias{simulate_track}
\title{Simulate tracks using different models}
\usage{
simulate_track(data, nsim = NULL, model = NULL)
}
\arguments{
\item{data}{A \code{track} R object, which is a list consisting of two elements:
\itemize{
\item \strong{\code{Trajectories}}: A list of interpolated trajectories, where each trajectory is a series of midpoints between consecutive footprints.
\item \strong{\code{Footprints}}: A list of data frames containing footprint coordinates, metadata (e.g., image reference, ID), and a marker indicating whether the footprint is actual or inferred.
}}

\item{nsim}{The number of simulations to run. Defaults to \code{1000} if not specified.}

\item{model}{The type of movement model to use. Options are \code{"Directed"}, \code{"Constrained"}, or \code{"Unconstrained"}. Defaults to \code{"Unconstrained"} if not provided.}
}
\value{
A \code{track simulation} R object consisting of a list of simulated trajectories stored as \code{track} R objects.
}
\description{
\code{simulate_track()} simulates movement trajectories based on an original set of tracks. Three movement models are available for simulation, each reflecting different levels of constraint in movement patterns. These models can represent biological or environmental constraints, such as movement along coastlines, rivers, or towards resources like water or food.
}
\details{
This function simulates movement trajectories based on the following models:
\itemize{
\item \strong{Directed}: This model simulates movement that follows a specific direction navigating with a compass (i.e., a directed walk/allothetic directed walk/oriented path) (Cheung et al., 2007, 2008). The trajectory is constrained by both the angular and linear properties of the original track, with minor deviations allowed to reflect natural variability.
\itemize{
\item \strong{Angular constraints}: The trajectory closely follows a specific direction, maintaining the overall angular orientation of the original track. Deviations of consecutive steps are governed by the angular standard deviation calculated from the original track using \code{TrajAngles()}.
\item \strong{Linear constraints}: Step lengths are constrained to the mean of the original track's step lengths, with variability allowed according to the standard deviation of step lengths computed with \code{TrajStepLengths()}.
\item \strong{Starting direction}: Fixed to the original direction (overall angular orientation) of the track.
}
}

This model is ideal for simulating movement directed toward a specific resource or constrained by natural barriers, with a relatively fixed direction and minor deviations.
\itemize{
\item \strong{Constrained}: This model simulates movement that correspond to a correllated random walk/idiothetic directed walk (Kareiva & Shigesada, 1983), corresponding to an animal navigating without a compass (Cheung et al., 2008), while still maintaining certain angular and linear characteristics of the original track. It provides more flexibility than the Directed model but is not entirely random like the Unconstrained model.
\itemize{
\item \strong{Angular constraints}: The trajectory does not follow a specific direction. Deviations of consecutive steps are governed by the angular standard deviation calculated from the original track using \code{TrajAngles()}.
\item \strong{Linear constraints}: Step lengths are constrained to the mean of the original track's step lengths, with variability allowed according to the standard deviation of step lengths computed with \code{TrajStepLengths()}.
\item \strong{Starting direction}: Fixed to the original direction (overall angular orientation) of the track.
}
}

This model is suitable for scenarios where movement is influenced by external constraints but allows for some degree of random exploration.
\itemize{
\item \strong{Unconstrained}: This model simulates movement that correspond to a correllated random walk/idiothetic directed walk (Kareiva & Shigesada, 1983), corresponding to an animal navigating without a compass (Cheung et al., 2008), while still maintaining certain angular and linear characteristics of the original track.
\itemize{
\item \strong{Angular constraints}: The trajectory does not follow a specific direction. Deviations of consecutive steps are governed by the angular standard deviation calculated from the original track using \code{TrajAngles()}.
\item \strong{Linear constraints}: Step lengths are constrained to the mean of the original track's step lengths, with variability allowed according to the standard deviation of step lengths computed with \code{TrajStepLengths()}.
\item \strong{Starting direction}: Randomly determined.
}
}

This model is suitable for simulating exploratory or dispersal behavior in open environments, where movement is random and not influenced by specific constraints.

Note: Simulations cannot be applied to trajectories with fewer than four steps as the standard deviations of angles and step lengths cannot be computed for such short trajectories. Consider using the \code{subset_track()} function to filter tracks with four or more steps.

The function utilizes the \pkg{trajr} package for key calculations:
\itemize{
\item \code{TrajGenerate()}: Generates a new trajectory based on random or directed movement models, constrained by specified parameters.
\item \code{TrajStepLengths()}: Calculates the step lengths (distances between consecutive points) of the original trajectory.
\item \code{TrajAngles()}: Computes the angles between consecutive segments of the trajectory, used to maintain directional movement in constrained models.
\item \code{TrajRotate()}: Rotates the trajectory by a specified angle to match the original direction or a random angle for unconstrained models.
\item \code{TrajTranslate()}: Translates the simulated trajectory to start at the same geographic location as the original.
}

The \code{NISTdegTOradian()} function from the \pkg{NISTunits} package is used to convert angles from degrees to radians.
}
\section{Logo}{

\if{html}{\figure{Logo.png}{options: width=30\%}}
}

\examples{
# Example 1: Simulate tracks using data from the Paluxy River
# Default model (Unconstrained movement)
simulated_tracks <- simulate_track(PaluxyRiver, nsim = 3)

# Example 2: Simulate tracks using the "Directed" model, representing movement
# toward a resource (e.g., water source)
simulated_tracks_directed <- simulate_track(PaluxyRiver, nsim = 3, model = "Directed")

# Example 3: Simulate tracks using the "Constrained" model, representing movement
# along a geographic feature (e.g., coastline)
simulated_tracks_constrained <- simulate_track(PaluxyRiver, nsim = 3, model = "Constrained")

# Example 4: Simulate tracks using the "Unconstrained" model (random exploratory movement)
simulated_tracks_unconstrained <- simulate_track(PaluxyRiver, nsim = 3, model = "Unconstrained")

# Subsetting trajectories with four or more steps in the MountTom dataset
sbMountTom <- subset_track(MountTom, tracks = c(1, 2, 3, 4, 7, 8, 9, 13, 15, 16, 18))

# Example 5: Simulate tracks using data from Mount Tom
# Default model (Unconstrained movement)
simulated_tracks_mt <- simulate_track(sbMountTom, nsim = 3)

# Example 6: Simulate tracks using the "Directed" model for Mount Tom, representing
# directed movement
simulated_tracks_mt_directed <- simulate_track(sbMountTom, nsim = 3, model = "Directed")

# Example 7: Simulate tracks using the "Constrained" model for Mount Tom, representing
# constrained movement
simulated_tracks_mt_constrained <- simulate_track(sbMountTom, nsim = 3, model = "Constrained")

# Example 8: Simulate tracks using the "Unconstrained" model for Mount Tom, representing
# random exploratory movement
simulated_tracks_mt_unconstrained <- simulate_track(sbMountTom, nsim = 3, model = "Unconstrained")

}
\references{
Cheung, A., Zhang, S., Stricker, C., & Srinivasan, M. V. (2007). Animal navigation: the difficulty of moving in a straight line. Biological cybernetics, 97, 47-61.

Cheung, A., Zhang, S., Stricker, C., & Srinivasan, M. V. (2008). Animal navigation: general properties of directed walks. Biological cybernetics, 99, 197-217.
}
\seealso{
\code{\link{tps_to_track}}, \code{\link{plot_sim}}, \code{\link{subset_track}}, \code{\link[trajr]{TrajGenerate}}, \code{\link[trajr]{TrajStepLengths}}, \code{\link[trajr]{TrajAngles}}, \code{\link[trajr]{TrajRotate}}, \code{\link[trajr]{TrajTranslate}}, \code{\link[NISTunits]{NISTdegTOradian}}
}
\author{
Humberto G. Ferrón

humberto.ferron@uv.es

Macroevolution and Functional Morphology Research Group (www.macrofun.es)

Cavanilles Institute of Biodiversity and Evolutionary Biology

Calle Catedrático José Beltrán Martínez, nº 2

46980 Paterna - Valencia - Spain

Phone: +34 (9635) 44477
}
