#' Simulate Gamma Dynamic GLM
#'
#' Generates a dynamic regression dataset for gamma family GLM
#' where the rate parameter follows a dynamic process.
#'
#' @param N Integer > 1, number of observations.
#' @param p Integer > 0, number of predictors.
#' @param alpha_gamma Numeric > 0, shape parameter for gamma distribution (default = 2).
#'
#' @return List with components:
#' \item{X}{N x p design matrix.}
#' \item{Y}{Numeric vector of length N, gamma distributed observations.}
#' \item{beta}{True coefficient vector.}
#' \item{alpha_gamma}{Shape parameter used.}
#'
#' @examples
#' set.seed(3)
#' dat <- simGamma(N = 500, p = 3)
#' hist(dat$Y, main = "Gamma Data")
#'
#' @export
simGamma <- function(N = 1000L, p = 4L, alpha_gamma = 2) {
  # Input checks
  if (N <= 1) stop("N must be > 1")
  if (p <= 0) stop("p must be > 0")
  if (alpha_gamma <= 0) stop("alpha_gamma must be > 0")
  
  # Generate design matrix
  X <- matrix(stats::rnorm(N * p), nrow = N, ncol = p)
  
  # Generate coefficients
  beta <- stats::rnorm(p, mean = 0, sd = 0.5)
  
  # Calculate linear predictor and rate parameter
  eta <- X %*% beta
  rate <- exp(eta)
  
  # Generate gamma responses
  Y <- stats::rgamma(N, shape = alpha_gamma, rate = rate)
  
  list(X = X, Y = Y, beta = beta, alpha_gamma = alpha_gamma)
}
