#' Plot original vs permuted time ordering
#'
#' Produces a side-by-side plot comparing the temporal component of an original
#' spatio-temporal point pattern with that of a permuted (or block-permuted)
#' version. This graphical diagnostic is intended to assess the effect of
#' temporal reordering procedures used in separability tests.
#'
#' The function is commonly employed to visualize temporal permutations
#' generated by procedures such as \code{\link{sim.procedures}} or
#' \code{\link{block.permut}}, which underpin pure and block permutations-based
#' inference for first-order separability (see Ghorbani et al., 2021,
#' Section 3.2).
#'
#' @param original A matrix or data frame with at least three columns \eqn{(x,y,t)}.
#'   The time coordinate is taken from column 3.
#' @param permuted A matrix or data frame with the same structure as \code{original}.
#'   The time coordinate is taken from column 3.
#' @param title Character string; title for the permuted panel.
#' @param col Character vector of length 2 giving colors for the original and permuted panels.
#' @param pch Plotting character passed to \code{\link[graphics]{plot}}.
#' @param ... Further graphical parameters passed to \code{\link[graphics]{plot}}.
#'
#' @details
#' The function uses base R graphics to display two panels:
#' \enumerate{
#'   \item The temporal ordering of the original point pattern.
#'   \item The temporal ordering after permutation or block permutation.
#' }
#'
#' This diagnostic is particularly useful when validating permutation-based
#' inference procedures such as \code{\link{chi2.test}},
#' \code{\link{dHS.test}}, and \code{\link{global.envelope.test}}.
#'
#' @return
#' The function is invoked for its side effect of producing a plot and
#' returns no value.
#'
#' @importFrom graphics par plot
#'
#' @seealso
#' \code{\link{sim.procedures}},
#' \code{\link{block.permut}},
#' \code{\link{dHS.test}},
#' \code{\link{chi2.test}}
#'
#' @references
#' Ghorbani, M., Vafaei, N., Dvořák, J., & Myllymäki, M. (2021).
#' Testing the first-order separability hypothesis for spatio-temporal point
#' patterns. \emph{Computational Statistics & Data Analysis}, \bold{161}, 107245.
#'
#' @author
#' Nafiseh Vafaei \email{nafiseh.vafaei@slu.se}
#'
#' @examples
#'
#' set.seed(123)
#' X <- cbind(x = runif(20), y = runif(20),
#'            time = seq(0, 1, length.out = 20))
#'
#' # Example: visualize pure permutation
#' sim_pure <- sim.procedures(X, nperm = 1, method = "pure")[[1]]
#' plot_procedures(X, sim_pure, title = "Pure Permutation")
#'
#' # Example: visualize block permutation (requires Block_permutation)
#' if (requireNamespace("combinat", quietly = TRUE)) {
#'   sim_block <- block.permut(nblocks = 4, X = X, nperm = 1)[[1]]
#'   plot_procedures(X, sim_block, title = "Block Permutation")
#' }
#'
#' @export
plot_procedures <- function(original, permuted,
                            title = "Permutation",
                            col = c("blue", "red"),
                            pch = 1,
                            ...) {

  # ---- validate ----
  if (!is.matrix(original) && !is.data.frame(original)) stop("`original` must be a matrix or data.frame.")
  if (!is.matrix(permuted) && !is.data.frame(permuted)) stop("`permuted` must be a matrix or data.frame.")

  original <- as.matrix(original)
  permuted <- as.matrix(permuted)

  if (ncol(original) < 3L) stop("`original` must have at least 3 columns (x, y, t).")
  if (ncol(permuted) < 3L) stop("`permuted` must have at least 3 columns (x, y, t).")
  if (nrow(original) != nrow(permuted)) stop("`original` and `permuted` must have the same number of rows.")
  if (!is.numeric(original[, 3]) || !is.numeric(permuted[, 3])) stop("Time column (column 3) must be numeric in both inputs.")
  if (!is.character(col) || length(col) != 2L) stop("`col` must be a character vector of length 2.")

  op <- graphics::par(mfrow = c(1, 2))
  on.exit(graphics::par(op), add = TRUE)

  graphics::plot(original[, 3], type = "o", col = col[1], pch = pch,
                 main = "Original time", ylab = "Time", xlab = "Index", ...)
  graphics::plot(permuted[, 3], type = "o", col = col[2], pch = pch,
                 main = title, ylab = "Time", xlab = "Index", ...)

  invisible(NULL)
}
