#' Filter a time series to reduce the short-memory dynamics using a low-order ARFIMA model
#'
#' Fits ARFIMA(p, d, q) models with \eqn{p, q \in \{0, 1\}} using
#' \code{\link[fracdiff]{fracdiff}}, selects the specification by AIC,
#' and applies the corresponding linear filter to the series.
#'
#' @param x A numeric vector containing the time series to be filtered.
#' @param n An integer giving the length of the filtered series. If \code{NULL},
#'   the length of \code{x} is used.
#'
#' @details
#' The function considers ARFIMA(1, d, 1), ARFIMA(1, d, 0), ARFIMA(0, d, 1),
#' and ARFIMA(0, d, 0) models, computes AIC for each, and selects the model
#' with the smallest AIC. If the absolute value of the selected AR or MA
#' coefficient is greater than or equal to 0.99, the procedure selects
#' a pure fractional model ARFIMA(0, d, 0). The input series is centered
#' before estimation.
#'
#' @return A numeric vector of length \code{n} containing the filtered series.
#'
#' @importFrom fracdiff fracdiff
#' @export
#'
#' @examples
#' set.seed(123)
#' sim <- fracdiff::fracdiff.sim(
#' n = 1000,
#' ar = 0.3,     # AR(1) coefficient
#' ma = -0.4,    # MA(1) coefficient
#' d  = 0.25     # fractional differencing parameter
#' )
#' x <- sim$series
#' y <- filterx(x)

filterx <- function(x, n = NULL) {
  if (is.null(n)) n <- length(x)
  x<-x-mean(x)

  est11 <- suppressWarnings(fracdiff(x, nar = 1, nma = 1, ar = 0.01, ma = 0.02))
  aic11 <- est11$log.likelihood * (-2) + 2 * 2
  opt_aic <- aic11
  armac <- c(est11$ar, est11$ma)
  opt_coef <- c(est11$ar, est11$ma, est11$d)

  est10 <- suppressWarnings(fracdiff(x, nar = 1, nma = 0, ar = 0))
  aic10 <- est10$log.likelihood * (-2) + 2 * 1
  if (aic10 < opt_aic) {
    armac <- c(est10$ar, 0)
    opt_aic <- aic10
    opt_coef <- c(est10$ar, 0, est10$d)
  }

  est01 <- suppressWarnings(fracdiff(x, nar = 0, nma = 1, ma = 0))
  aic01 <- est01$log.likelihood * (-2) + 2 * 1
  if (aic01 < opt_aic) {
    armac <- c(0, est01$ma)
    opt_aic <- aic01
    opt_coef <- c(0, est01$ma, est01$d)
  }

  est00 <- suppressWarnings(fracdiff(x, nar = 0, nma = 0))
  aic00 <- est00$log.likelihood * (-2) + 2 * 0
  if (aic00 < opt_aic) {
    armac <- c(0, 0)
    opt_aic <- aic00
    opt_coef <- c(0, 0, est00$d)
  }

  if (abs(armac[1]) >= 0.99) {
    armac[1] <- 0
    armac[2] <- 0
    opt_coef <- c(0, 0, est00$d)
  }
  if (abs(armac[2]) >= 0.99) {
    armac[1] <- 0
    armac[2] <- 0
    opt_coef <- c(0, 0, est00$d)
  }

  cvec <- coefs(armac[1], armac[2], n)
  newx <- x
  for (i in 1:n)
    newx[i] <- t(rev(x[1:i])) %*% cvec[1:i]

  return(newx)
}

# Compute filter coefficients
#' @keywords internal
#' @noRd
coefs <- function(car, cma, n) {
  coef1 <- rep(1, n)
  coef1[1] <- 0
  coef1[3:n] <- rep(cma, (n - 2))
  coef1[2:n] <- cumprod(coef1[2:n])
  coef2 <- cma - car
  coef <- coef1 * coef2
  coef[1] <- 1
  return(coef)
}
