% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hist_med.R
\name{histmed}
\alias{histmed}
\title{Wasserstein Median of Histograms}
\usage{
histmed(hists, weights = NULL, L = 2000L, ...)
}
\arguments{
\item{hists}{a length-\eqn{N} list of histograms (\code{"histogram"} objects)
with identical \code{breaks}.}

\item{weights}{a weight for each histogram; if \code{NULL} (default), uniform
weights are used. Otherwise, it should be a length-\eqn{N} vector of
nonnegative weights.}

\item{L}{number of quantile levels used to approximate the median
(default: 2000). Larger \code{L} gives a more accurate approximation at
increased computational cost.}

\item{...}{extra parameters including \describe{
  \item{abstol}{stopping criterion for iterations (default: 1e-8).}
  \item{maxiter}{maximum number of iterations (default: 496).}
  \item{print.progress}{logical; whether to show current iteration
       (default: \code{FALSE}).}
}}
}
\value{
a \code{"histogram"} object representing the Wasserstein median.
}
\description{
Given multiple histograms represented as \code{"histogram"} S3 objects with
common breaks, compute their Fréchet (geometric) median under the
2-Wasserstein distance. In 1D, this is implemented by mapping histograms
to their quantile functions and running a Weiszfeld-type algorithm for 
the geometric median in the Hilbert space of quantile
functions.
}
\examples{
\donttest{
#----------------------------------------------------------------------
#                      Binned from Two Gaussians
#
# Generate 12 histograms from N(-4,1/4) and 8 from N(4,1/4)
#----------------------------------------------------------------------
# COMMON SETTING
set.seed(100)
bk = seq(from=-10, to=10, length.out=20)
n_signal = 12

# GENERATE HISTOGRAMS WITH COMMON BREAKS
hist_all = list()
for (i in 1:n_signal){
  hist_all[[i]] = hist(stats::rnorm(200, mean=-4, sd=0.5), breaks=bk)
}
for (j in (n_signal+1):20){
  hist_all[[j]] = hist(stats::rnorm(200, mean=+4, sd=0.5), breaks=bk)
}

# COMPUTE THE BARYCENTER AND THE MEDIAN 
h_bary = histbary(hist_all)
h_med  = histmed(hist_all)

# VISUALIZE
xt   <- round(h_med$mids, 1) 
opar <- par(no.readonly=TRUE)
par(mfrow=c(1,3), pty="s")
barplot(hist_all[[1]]$density, col=rgb(0,0,1,1/4), 
        ylim=c(0, 0.75), main="Two Types", names.arg=xt)
barplot(hist_all[[20]]$density, col=rgb(1,0,0,1/4), 
        ylim=c(0, 0.75), add=TRUE)
barplot(h_med$density,  names.arg=xt, main="Median", ylim=c(0, 0.75))
barplot(h_bary$density, names.arg=xt, main="Barycenter", ylim=c(0, 0.75))
par(opar)
}

}
\concept{histogram}
