% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fittestArima.r
\name{fittestArima}
\alias{fittestArima}
\title{Automatic ARIMA fitting, prediction and accuracy evaluation}
\usage{
fittestArima(
  timeseries,
  timeseries.test = NULL,
  h = NULL,
  na.action = stats::na.omit,
  level = c(80, 95),
  ...
)
}
\arguments{
\item{timeseries}{A vector or univariate time series which contains the
values used for fitting an ARIMA model.}

\item{timeseries.test}{A vector or univariate time series containing a
continuation for \code{timeseries} with actual values. It is used as a
testing set and base for calculation of prediction error measures. Ignored
if \code{NULL}.}

\item{h}{Number of consecutive values of the time series to be predicted. If
\code{h} is \code{NULL}, the number of consecutive values to be predicted is
assumed to be equal to the length of \code{timeseries.test}. Required when
\code{timeseries.test} is \code{NULL}.}

\item{na.action}{A function for treating missing values in \code{timeseries}
and \code{timeseries.test}. The default function is
\code{\link[stats]{na.omit}}, which omits any missing values found in
\code{timeseries} or \code{timeseries.test}.}

\item{level}{Confidence level for prediction intervals.}

\item{...}{Additional arguments passed to the
\code{\link[forecast]{auto.arima}} modelling function.}
}
\value{
A list with components: \item{model}{A list of class "ARIMA"
containing the best fitted ARIMA model. See the \code{\link[forecast]{auto.arima}}
function in the \code{forecast} package.} \item{parameters}{A list
containing the parameters of the best fitted ARIMA model. See the
\code{\link{arimaparameters}} function.} \item{AICc}{Numeric value of the
computed AICc criterion of the fitted model.} \item{AIC}{Numeric value of
the computed AIC criterion of the fitted model.} \item{BIC}{Numeric value of
the computed BIC criterion of the fitted model.} \item{logLik}{Numeric value
of the computed log-likelihood of the fitted model.} \item{pred}{A list with
the components \code{mean}, \code{lower} and \code{upper}, containing the
predictions and the lower and upper limits for prediction intervals,
respectively. All components are time series. See the \code{\link[forecast]{forecast}}
function in the \code{forecast} package.} \item{MSE}{Numeric value of the
resulting MSE error of prediction.} \item{NMSE}{Numeric value of the
resulting NMSE error of prediction.} \item{MAPE}{Numeric value of the
resulting MAPE error of prediction.} \item{sMAPE}{Numeric value of the
resulting sMAPE error of prediction.} \item{MaxError}{Numeric value of the
maximal error of prediction.}
}
\description{
The function predicts and returns the next n consecutive values of a
univariate time series using an automatically best fitted ARIMA model. It
also evaluates the fitness of the produced model, using AICc, AIC, BIC and
logLik criteria, and its prediction accuracy, using the MSE, NMSE, MAPE,
sMAPE and maximal error accuracy measures.
}
\details{
The ARIMA model is automatically fitted by the
\code{\link[forecast]{auto.arima}} function and it is used for prediction by
the \code{\link[forecast]{forecast}} function both in the \code{forecast}
package.

The fitness criteria AICc, AIC (\code{\link{AIC}}), BIC (\code{\link{BIC}})
and log-likelihood (\code{\link{logLik}}) are extracted from the fitted
ARIMA model. Also, the prediction accuracy of the model is computed by means
of MSE (\code{\link{MSE}}), NMSE (\code{\link{NMSE}}), MAPE
(\code{\link{MAPE}}), sMAPE (\code{\link{sMAPE}}) and maximal error
(\code{\link{MAXError}}) measures.
}
\examples{

\donttest{
data(CATS,CATS.cont)
fArima <- fittestArima(CATS[,1],CATS.cont[,1])
#predicted values
pred <- fArima$pred$mean
#model information
cbind(AICc=fArima$AICc, AIC=fArima$AIC, BIC=fArima$BIC,
 logLik=fArima$logLik, MSE=fArima$MSE, NMSE=fArima$NMSE,
 MAPE=fArima$MSE, sMAPE=fArima$MSE, MaxError=fArima$MaxError)

#plotting the time series data
plot(c(CATS[,1],CATS.cont[,1]),type='o',lwd=2,xlim=c(960,1000),ylim=c(0,200),
 xlab="Time",ylab="ARIMA")
#plotting the predicted values
lines(ts(pred,start=981),lwd=2,col='blue')
#plotting prediction intervals
lines(ts(fArima$pred$upper[,2],start=981),lwd=2,col='light blue')
lines(ts(fArima$pred$lower[,2],start=981),lwd=2,col='light blue')
}

}
\references{
R.J. Hyndman and G. Athanasopoulos, 2013, Forecasting:
principles and practice. OTexts.

R.H. Shumway and D.S. Stoffer, 2010, Time Series Analysis and Its
Applications: With R Examples. 3rd ed. 2011 edition ed. New York, Springer.
}
\seealso{
\code{\link{fittestArimaKF}}, \code{\link{fittestLM}},
\code{\link{marimapred}}
}
\author{
Rebecca Pontes Salles
}
\keyword{ARIMA}
\keyword{adjustment}
\keyword{automatic}
\keyword{criterion}
\keyword{errors}
\keyword{evaluation}
\keyword{fitting}
\keyword{prediction}
