#' @title Convex check
#' @description This function checks if the given game is convex.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param instance A logical value. By default, \code{instance=FALSE}.
#' @return \code{TRUE} if the game is convex, \code{FALSE} otherwise. If \code{instance=TRUE} and the game is not convex, the function also returns the positions (binary order positions if \code{binary=TRUE}; lexicographic order positions otherwise) of a pair of coalitions violating the Zumsteg convexity characterization.
#' @details A game \eqn{v\in G^N} is convex if \eqn{v(S \cap T) + v(S \cup T) \ge v(S)+v(T)} for all
#' \eqn{S,T \in 2^N}. Zumsteg, S. (1995) shows that \eqn{v} is convex if \eqn{v(S \cup {i}\cup {j}) + v(S) \ge v(S\cup {i})+v(S\cup {j})} for all
#' \eqn{S\in 2^N} and \eqn{i,j\in N\backslash S} such that \eqn{i\not=j}.
#'
#' A game \eqn{v\in G^N} is concave if \eqn{-v} is convex.
#' @examples
#' v1 <- c(5, 2, 2, 1, 8, 8, 6, 4, 3, 3, 12, 10, 10, 6, 14)
#' convexcheck(v1)
#' v2 <- c(0, 0, 0, 2, 2, 1, 3)
#' convexcheck(v2, binary = FALSE, instance = TRUE)
#'
#' # How to check if a game is concave:
#' v.conc <- c(4, 3, 3, 2, 6, 6, 5, 5, 4, 4, 7, 6, 6, 6, 7) # concave game
#' convexcheck(-v.conc)
#' @references Zumsteg, S. (1995). \emph{Non-cooperative aspects of cooperative game theory and related computational problems}. PhD thesis, ETH Zurich.
#' @seealso \link{strategicallyequivalentcheck}, \link{superadditivecheck}
#' @export

convexcheck <- function(v, binary = FALSE, instance = FALSE) {

  ################################
  ### Comprobación datos entrada###
  ################################

  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }
  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  ################################
  ##### Creación de variables######
  ################################

  checkR <- TRUE # Por defecto, mientras no encuentre un contraejemplo, checkR=TRUE
  exampleR <- NULL # Por defecto, no hay contraejemplo
  S <- 0 # Inicializo coalición S.
  v <- c(0, v) # Añadimos el 0 de la coalición vacía.
  tol <- 100*.Machine$double.eps

  ################################
  ###### Cuerpo de la función######
  ################################

  while (checkR & S <= nC - 1) # Mientras no haya encontrado un contraejemplo y no haya superado el número máximo de coaliciones, hago la comprobación.
  {
    vS <- v[S + 1] # Valor de v(S)
    a <- 0 # Inicio un contador para poder eliminar jugadores de la coalición.
    # Calculamos los jugadores pertenecientes a la complementaria:
    num <- nC - S
    J <- floor(log2(num)) + 1
    Sc <- c() # Inicializo la coalición S

    for (ii in 1:J)
    {
      if (num %% 2 == 1) {
        Sc <- c(Sc, ii)
      }
      num <- num %/% 2
    }
    Sc_ultimo <- Sc[-length(Sc)]
    if (length(Sc) > 1) # Si hay solo un jugador no hacemos la comprobación para evitar problemas con los bucles.
      {
        for (ii in Sc_ultimo)
        {
          Sii <- S + 2^(ii - 1) # Coalición S U i.
          vSii <- v[Sii + 1] # Valor de la coalición S U i.
          a <- a + 1 # Determino el lugar hasta el que tengo que eliminar jugadores para después añadirlos a S.
          Resto <- Sc
          Resto <- Resto[-(1:a)] # En resto están los jugadores distintos de i que puedo añadir a S.
          for (jj in Resto)
          {
            Sjj <- S + 2^(jj - 1)
            vSjj <- v[Sjj + 1] # Coalición S U j y valor
            Sij <- Sii + 2^(jj - 1)
            vSij <- v[Sij + 1] # Coalición S U i U j y valor
            if (((vSij + vS) - (vSii + vSjj)) + tol < 0) # Condiciones de convexidad de Zumsteg
              {
                checkR <- FALSE
                exampleR <- c(Sii, Sjj)
              }
          }
        }
      }
    S <- S + 1
  }

  ################################
  ###### Salidas de la función#####
  ################################


  if (instance == FALSE) { # Si el usuario sólo pide una salida, es decir, instance=FALSE
    return(check = checkR)
  } else { # Si instance=TRUE, la función devuelve también las posiciones de las coaliciones S y R.
    if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, devolvemos las posiciones lexicográficas de un par de coaliciones que falla.
      return(list(check = checkR, example = c(codebin2lex(n, exampleR[1]), codebin2lex(n, exampleR[2]))))
    } else { # Si el juego se introdujo en binario, devolvemos las posiciones binarias de un par de coaliciones que falla.
      return(list(check = checkR, example = exampleR))
    }
  }
} # Fin de la función
