% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/binspp_auxiliary.R
\name{rThomasInhom}
\alias{rThomasInhom}
\title{Simulate a realization of Thomas-type cluster point process with complex inhomogeneities}
\usage{
rThomasInhom(
  kappa,
  alpha,
  omega,
  W,
  W_dil,
  betavec = NULL,
  alphavec = NULL,
  omegavec = NULL,
  z_beta = NULL,
  z_alpha = NULL,
  z_omega = NULL
)
}
\arguments{
\item{kappa}{intensity or intensity function of the parent process, scalar or pixel image object of class \code{\link[spatstat.geom:im]{spatstat.geom::im()}} from the \pkg{spatstat} package.}

\item{alpha}{scalar, influences the mean number of points in individual clusters, see Details.}

\item{omega}{scalar, influences the spread of individual clusters, see Details.}

\item{W}{the observation window where the realization is to be generated, in the \cr \code{\link[spatstat.geom:owin]{spatstat.geom::owin()}} format of the \pkg{spatstat} package.}

\item{W_dil}{the observation window dilated by the assumed maximal cluster radius, as a binary mask with the same resolution as the covariates.}

\item{betavec}{vector of parameters describing the dependence of the intensity function of the parent process on covariates in the list \emph{z_beta}.}

\item{alphavec}{vector of parameters describing the dependence of the mean number of points in a cluster on covariates in the list \emph{z_alpha}.}

\item{omegavec}{vector of parameters describing the dependence of the spread of the clusters on covariates in the list \emph{z_omega}.}

\item{z_beta}{list of covariates describing the intensity function of the parent process, each covariate being a pixel image as used in the \pkg{spatstat} package.}

\item{z_alpha}{list of covariates describing the location-dependent mean number of points in a cluster, each covariate being a pixel image as used in the \pkg{spatstat} package.}

\item{z_omega}{list of covariates describing the location-dependent scale of a cluster, each covariate being a pixel image as used in the \pkg{spatstat} package.}
}
\value{
A planar point pattern, object of the type \code{\link[spatstat.geom:ppp]{spatstat.geom::ppp()}}
used in the \pkg{spatstat} package.
}
\description{
The means to simulate realizations
from the Thomas-type cluster point process with complex
inhomogeneities are provided.
}
\details{
A realization of a Thomas-type cluster
point process model is produced with possible inhomogeneity
(described by covariates) in any or all of the following model
components: intensity function of the parent process, mean number
of points in a cluster, scale of the clusters.
Model parametrization is described in the documentation to the
function \code{\link[=estintp]{estintp()}}. The parent process is generated using the function
\code{\link[spatstat.random:rpoispp]{spatstat.random::rpoispp()}} from the \pkg{spatstat.random} package in the dilated
observation window \emph{W_dil} to avoid edge-effects. The offspring points
are generated directly from the appropriate normal distributions. The resulting
point pattern is eventually clipped to the smaller
observation window \emph{W}.
}
\examples{

library(spatstat)
# Unit square observation window:
W <- owin()

# Dilation of the observation window:
W_dil <- dilation(W, 0.1)
W_dil <- as.mask(W_dil)

# Define covariates:
f1 <- function(x, y) { x }
f2 <- function(x, y) { y }
f3 <- function(x, y) { 1 - (y - 0.5) ^ 2 }
cov1 <- as.im(f1, W = W_dil)
cov2 <- as.im(f2, W = W_dil)
cov3 <- as.im(f3, W = W_dil)


# Stationary Thomas process:
X <- rThomasInhom(kappa = 50, alpha = log(10), omega = log(0.01),
       W = W, W_dil = W_dil)
plot(X)


# Thomas-type cluster process with inhomogeneity in all model components:
X <- rThomasInhom(kappa = 10, betavec = c(1), z_beta = list(cov1),
            alpha = log(10), alphavec = c(1), z_alpha = list(cov2),
            omega = log(0.01), omegavec = c(1), z_omega = list(cov3),
            W = W, W_dil = W_dil)
plot(X)

}
