% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bm_RangeSize.R
\name{bm_RangeSize}
\alias{bm_RangeSize}
\alias{bm_RangeSize,data.frame,data.frame-method}
\alias{bm_RangeSize,SpatRaster,SpatRaster-method}
\title{Analyze the range size differences between projections of species distribution models}
\usage{
bm_RangeSize(proj.current, proj.future, ordinal = FALSE)

\S4method{bm_RangeSize}{data.frame,data.frame}(proj.current, proj.future, ordinal = FALSE)

\S4method{bm_RangeSize}{SpatRaster,SpatRaster}(proj.current, proj.future, ordinal = FALSE)
}
\arguments{
\item{proj.current}{a \code{data.frame}, \code{\link[raster:stack]{RasterLayer}} 
or \code{\link[terra:rast]{SpatRaster}} object containing the initial binary projection(s) 
of the (ensemble) species distribution model(s)}

\item{proj.future}{a \code{data.frame}, \code{\link[raster:stack]{RasterLayer}} 
or \code{\link[terra:rast]{SpatRaster}} object containing the final binary projection(s) 
of the (ensemble) species distribution model(s)}

\item{ordinal}{a \code{logical} indicating whether or not the projections should be considered as ordinal data}
}
\value{
A \code{list} containing two objects :
\describe{
  \item{Compt.By.Species}{a \code{data.frame} containing the summary of range change for each 
  comparison
  \itemize{
    \item \code{Loss} : number of pixels predicted to be lost
    \item \code{Stable_Abs} : number of pixels not currently occupied and not predicted to be
    \item \code{Stable_Pres} : number of pixels currently occupied and predicted to remain 
    occupied
    \item \code{Gain} : number of pixels predicted to be gained
    \item \code{PercLoss} : percentage of pixels currently occupied and predicted to be lost 
    (\code{Loss / (Loss + Stable_Pres)})
    \item \code{PercGain} : percentage of pixels predicted to be gained compare to the 
    number of pixels currently occupied (\code{Gain / (Loss + Stable_Pres)})
    \item \code{SpeciesRangeChange} : percentage of pixels predicted to change (loss or gain) 
    compare to the number of pixels currently occupied (\code{PercGain - PercLoss})
    \item \code{CurrentRangeSize} : number of pixels currently occupied
    \item \code{FutureRangeSize0Disp} : number of pixels predicted to be occupied, assuming 
    no migration
    \item \code{FutureRangeSize1Disp} : number of pixels predicted to be occupied, assuming 
    migration
  }
  }
  \item{loss.gain}{an object in the same form than the input data (\code{proj.current} and 
  \code{proj.future}) and containing a value for each point/pixel of each comparison among :
  \itemize{
    \item \code{-2} : predicted to be lost
    \item \code{-1} : predicted to remain occupied
    \item \code{0} : predicted to remain unoccupied
    \item \code{1} : predicted to be gained
  }
  }
  \item{Diff.By.Pixel}{an object in the same form than the input data (\code{proj.current} and 
  \code{proj.future}) and containing a value for each point/pixel of each comparison obtain with :
  \itemize{
  \item \code{Future - 2* Current} for binary data
  \item \code{Future - Current} for nonbinary
  }
  }
}
}
\description{
This function allows to calculate the absolute number of locations (pixels) lost, 
stable and gained, as well as the corresponding relative proportions, between two (or more) 
binary projections of (ensemble) species distribution models (\emph{which can represent new 
time scales or environmental scenarios for example}).
}
\details{
Note that \bold{this function is only relevant to compare binary projections, made on the 
same area with the same resolution}. \cr

Comparison between \code{proj.current} and \code{proj.future} depends 
on the number of projection in both objects:
| \code{proj.current}   | \code{proj.future} | \bold{Comparison}  |
| ------------------------- | ---------------------- | --------------------  |
| \bold{1 projection} (\emph{e.g. data.frame with 1 column, SpatRaster with 1 layer}) | \bold{1 projection}  (\emph{e.g. data.frame with 1 column, SpatRaster with 1 layer})  | comparison of both projection  (\emph{e.g. current vs future conditions for the same model ; current vs current condition for two different models}) |
| \bold{\code{n} projections}  (\emph{e.g. data.frame with n column, SpatRaster with n layer}) |  \bold{\code{n} projections}  (\emph{e.g. data.frame with n column, SpatRaster with n layer}) |  comparing projection \code{i} in \code{proj.current} to projection \code{i} in \code{proj.future}  (\emph{e.g. comparing current vs future condition for n models}) |
| \bold{\code{1} projection}   (\emph{e.g. data.frame with 1 column, SpatRaster with 1 layer}) |  \bold{\code{n} projections}  (\emph{e.g. data.frame with n column, SpatRaster with n layer}) |  comparing projection in \code{proj.current} to each projection in \code{proj.future}  (\emph{e.g. comparing current vs n different future condition (e.g. climate change scenario) for 1 model}) |

For binary data,\code{Diff.By.Pixel} object is obtained by applying the simple following formula :
\deqn{proj.future - 2 * proj.current}  

For non binary data, \code{Diff.By.Pixel} object is obtained by rescaling the two projection to a 0 to 1 scale
and applying the simple following formula :
\deqn{proj.future - proj.current}
}
\examples{
library(terra)

# Load species occurrences (6 species available)
data(DataSpecies)
head(DataSpecies)

# Select the name of the studied species
myRespName <- 'GuloGulo'

# Get corresponding presence/absence data
myResp <- as.numeric(DataSpecies[, myRespName])

# Get corresponding XY coordinates
myRespXY <- DataSpecies[, c('X_WGS84', 'Y_WGS84')]

# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
data(bioclim_current)
myExpl <- terra::rast(bioclim_current)

\dontshow{
myExtent <- terra::ext(0,30,45,70)
myExpl <- terra::crop(myExpl, myExtent)
}

# --------------------------------------------------------------- #
file.out <- paste0(myRespName, "/", myRespName, ".AllModels.models.out")
if (file.exists(file.out)) {
  myBiomodModelOut <- get(load(file.out))
} else {

  # Format Data with true absences
  myBiomodData <- BIOMOD_FormatingData(resp.name = myRespName,
                                       resp.var = myResp,
                                       resp.xy = myRespXY,
                                       expl.var = myExpl)

  # Model single models
  myBiomodModelOut <- BIOMOD_Modeling(bm.format = myBiomodData,
                                      modeling.id = 'AllModels',
                                      models = c('RF', 'GLM'),
                                      CV.strategy = 'random',
                                      CV.nb.rep = 2,
                                      CV.perc = 0.8,
                                      OPT.strategy = 'bigboss',
                                      metric.eval = c('TSS', 'ROC'),
                                      var.import = 3,
                                      seed.val = 42)
}

models.proj <- get_built_models(myBiomodModelOut, algo = "RF")
  # Project single models
  myBiomodProj <- BIOMOD_Projection(bm.mod = myBiomodModelOut,
                                    proj.name = 'CurrentRangeSize',
                                    new.env = myExpl,
                                    models.chosen = models.proj,
                                    metric.binary = 'all',
                                    build.clamping.mask = TRUE)


# --------------------------------------------------------------- #
# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
data(bioclim_future)
myExplFuture <- terra::rast(bioclim_future)
\dontshow{
myExtent <- terra::ext(0,30,45,70)
myExplFuture <- terra::crop(myExplFuture, myExtent)
}
# Project onto future conditions
myBiomodProjectionFuture <- BIOMOD_Projection(bm.mod = myBiomodModelOut,
                                              proj.name = 'FutureRangeSize',
                                              new.env = myExplFuture,
                                              models.chosen = models.proj,
                                              metric.binary = 'TSS')

# Load current and future binary projections
CurrentProj <- get_predictions(myBiomodProj,
                               metric.binary = "TSS",
                               model.as.col = TRUE)
FutureProj <- get_predictions(myBiomodProjectionFuture,
                               metric.binary = "TSS",
                               model.as.col = TRUE)
# Compute differences
myBiomodRangeSize <- bm_RangeSize(proj.current = CurrentProj, proj.future = FutureProj)

myBiomodRangeSize$Compt.By.Models
plot(myBiomodRangeSize$Diff.By.Pixel)



}
\seealso{
\code{\link{BIOMOD_Projection}}, \code{\link{BIOMOD_EnsembleForecasting}}, 
\code{\link{bm_PlotRangeSize}}

Other Secondary functions: 
\code{\link{bm_BinaryTransformation}()},
\code{\link{bm_CrossValidation}()},
\code{\link{bm_FindOptimStat}()},
\code{\link{bm_MakeFormula}()},
\code{\link{bm_ModelAnalysis}()},
\code{\link{bm_ModelingOptions}()},
\code{\link{bm_PlotEvalBoxplot}()},
\code{\link{bm_PlotEvalMean}()},
\code{\link{bm_PlotRangeSize}()},
\code{\link{bm_PlotResponseCurves}()},
\code{\link{bm_PlotVarImpBoxplot}()},
\code{\link{bm_PseudoAbsences}()},
\code{\link{bm_RunModelsLoop}()},
\code{\link{bm_SRE}()},
\code{\link{bm_SampleBinaryVector}()},
\code{\link{bm_SampleFactorLevels}()},
\code{\link{bm_Tuning}()},
\code{\link{bm_VariablesImportance}()}
}
\author{
Wilfried Thuiller, Damien Georges, Bruno Lafourcade
}
\concept{Secondary functions}
\keyword{"species}
\keyword{change"}
\keyword{gain}
\keyword{loss}
\keyword{projections}
\keyword{range}
