% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/netclu_walktrap.R
\name{netclu_walktrap}
\alias{netclu_walktrap}
\title{Community structure detection via short random walks}
\usage{
netclu_walktrap(
  net,
  weight = TRUE,
  cut_weight = 0,
  index = names(net)[3],
  steps = 4,
  bipartite = FALSE,
  site_col = 1,
  species_col = 2,
  return_node_type = "both",
  algorithm_in_output = TRUE
)
}
\arguments{
\item{net}{The output object from \code{\link[=similarity]{similarity()}} or
\code{\link[=dissimilarity_to_similarity]{dissimilarity_to_similarity()}}. If a \code{data.frame} is used, the first two
columns represent pairs of sites (or any pair of nodes), and the next
column(s) are the similarity indices.}

\item{weight}{A \code{boolean} indicating if the weights should be considered
if there are more than two columns.}

\item{cut_weight}{A minimal weight value. If \code{weight} is TRUE, the links
between sites with a weight strictly lower than this value will not be
considered (0 by default).}

\item{index}{Name or number of the column to use as weight. By default,
the third column name of \code{net} is used.}

\item{steps}{The length of the random walks to perform.}

\item{bipartite}{A \code{boolean} indicating if the network is bipartite
(see Details).}

\item{site_col}{Name or number for the column of site nodes
(i.e. primary nodes).}

\item{species_col}{Name or number for the column of species nodes
(i.e. feature nodes).}

\item{return_node_type}{A \code{character} indicating what types of nodes
(\code{site}, \code{species}, or \code{both}) should be returned in the output
(\code{return_node_type = "both"} by default).}

\item{algorithm_in_output}{A \code{boolean} indicating if the original output
of \link[igraph:cluster_walktrap]{cluster_walktrap} should be returned in the
output (\code{TRUE} by default, see Value).}
}
\value{
A \code{list} of class \code{bioregion.clusters} with five slots:
\enumerate{
\item{\strong{name}: A \code{character} containing the name of the algorithm.}
\item{\strong{args}: A \code{list} of input arguments as provided by the user.}
\item{\strong{inputs}: A \code{list} of characteristics of the clustering process.}
\item{\strong{algorithm}: A \code{list} of all objects associated with the
clustering procedure, such as original cluster objects (only if
\code{algorithm_in_output = TRUE}).}
\item{\strong{clusters}: A \code{data.frame} containing the clustering results.}}

In the \code{algorithm} slot, if \code{algorithm_in_output = TRUE}, users can
find the output of
\link[igraph:cluster_walktrap]{cluster_walktrap}.
}
\description{
This function finds communities in a (un)weighted undirected network via
short random walks.
}
\details{
This function is based on random walks (Pons & Latapy, 2005)
as implemented in the \href{https://cran.r-project.org/package=igraph}{igraph}
package (\link[igraph:cluster_walktrap]{cluster_walktrap}).
}
\note{
Although this algorithm was not primarily designed to deal with bipartite
networks, it is possible to consider the bipartite network as unipartite
network (\code{bipartite = TRUE}).

Do not forget to indicate which of the first two columns is
dedicated to the site nodes (i.e. primary nodes) and species nodes (i.e.
feature nodes) using the arguments \code{site_col} and \code{species_col}.
The type of nodes returned in the output can be chosen with the argument
\code{return_node_type} equal to \code{both} to keep both types of nodes,
\code{sites} to preserve only the site nodes, and \code{species} to
preserve only the species nodes.
}
\examples{
comat <- matrix(sample(1000, 50), 5, 10)
rownames(comat) <- paste0("Site", 1:5)
colnames(comat) <- paste0("Species", 1:10)

net <- similarity(comat, metric = "Simpson")
com <- netclu_walktrap(net)

net_bip <- mat_to_net(comat, weight = TRUE)
clust2 <- netclu_walktrap(net_bip, bipartite = TRUE)

}
\references{
Pons P & Latapy M (2005) Computing Communities in Large Networks
Using Random Walks. In Yolum I, Güngör T, Gürgen F, Özturan C (eds.),
\emph{Computer and Information Sciences - ISCIS 2005}, Lecture Notes in
Computer Science, 284-293.
}
\seealso{
For more details illustrated with a practical example,
see the vignette:
\url{https://biorgeo.github.io/bioregion/articles/a4_3_network_clustering.html}.

Associated functions:
\link{netclu_infomap} \link{netclu_louvain} \link{netclu_oslom}
}
\author{
Maxime Lenormand (\email{maxime.lenormand@inrae.fr}) \cr
Pierre Denelle (\email{pierre.denelle@gmail.com}) \cr
Boris Leroy (\email{leroy.boris@gmail.com})
}
