## ------------------------------------------------------------------------
# 
# "boot_local(X,n,h,kernel,t)" 
# 
# TFT local bootstrap
#
# ------------------------------------------------------------------------
##


#' @aliases boot_local
#' @title TFT Local Bootstrap.
#' @description The local bootstrap is used to bootstrap the Fourier coefficients for 
#' the Time Frequency Toggle (TFT)-Bootstrap (see Kirch and Politis (2011)).
#' @param X A numeric vector representing a time series.
#' @param n An integer; by default is the length of time series \code{X} but allow for a 
#' smaller sample size m to perform moon bootstrap.
#' @param h A numeric value specifying the bandwidth used to compute the kernel estimator 
#' in case of local bootstrap. 
#' By default it is equal to \eqn{n^{-2/3}}.
#' @param kernel An integer value indicating the kernel type. Use \code{0} 
#' for the Daniell kernel or any other value for the Bartlett-Priestley (Epanechnikov) kernel 
#' (by default).
#' @param t An integer indicating the number of bootstrap replications.
#' @details 
#' The function first centers \code{X} by subtracting its mean, calculates the Fourier coefficients 
#' using the Fast Fourier Transform (FFT), and computes resampling probabilities based on the 
#' specified kernel \code{kernel} and bandwidth \code{h}. 
#' These probabilities are used to select random Fourier 
#' coefficients, which are then transformed back to generate standardized bootstrap replicates.
#' @return A matrix where each column contains a bootstrap replicate of the time series \code{X}.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Kirch, C. and Politis, D. N. (2011). TFT-Bootstrap: Resampling time series in the frequency
#'  domain to obtain replicates in the time domain, \emph{Annals of Statistics}, vol. 
#' @author We are grateful to Claudia Kirch for providing the original code in \code{R}.
#' @seealso \code{\link{tft_boot}}, \code{\link{boot_res}},
#' \code{\link{boot_wild}}.
#' @keywords Bootstrap "Frequency and time domain" "Fast Fourier Transform"
#' @export
#' @examples 
#' # see the mother function tft_boot


boot_local<-function(X,n=length(X),h,kernel,t){
  # t : number of bootstrap replicates
  # h a smoothing parameter (should be ravers small  
  # generates standardized bootstrap replicates of X-mean(X)
  #kernel=0: daniell; else: bartlett-priestley (epanechnikov)
  
  # test if n is even, else remove a point
  if (n %%2 ==1){ n=n-1
  X=X[2:n]
  }
  Z<-X-mean(X)
  N<-n/2-1  #n even
  
  prob<-get_prob(n,h,kernel) # h is a tuning parameter -- essentially a bandwidth
  wind<-trunc(n*h)
  prob2<-c(prob[(wind+1):2],prob)
  
  fourier<-fft(Z)/sqrt(n)
  #n even
  Refou<-Re(fourier[2:(n/2)])
  Imfou<-Im(fourier[2:(n/2)])
  fou<-c(Refou[(N-wind+2):N],0,Refou,0,Refou[1:(wind-1)],Imfou[(N-wind+2):N],0,Imfou,0,Imfou[1:(wind-1)])
  
  fou1<-c(Refou[(N-wind+2):N],0,Refou,0,Refou[1:(wind-1)])
  fou2<-c(Imfou[(N-wind+2):N],0,Imfou,0,Imfou[1:(wind-1)])
  
  erg<-array(rep(0,times=t*n),dim=c(n,t))
  
  for(i in 1:t){
    b<-(N+2*wind)*rbinom(2*N,1,0.5)
    s<-sample(1+2*wind,2*N,replace=TRUE,prob=prob2)
    neu2<-fou[b[1:N]+(0:(N-1))+s[1:N]]+1i*(fou[b[(N+1):(2*N)]+(0:(N-1))+s[(N+1):(2*N)]])
    
    erg[,i]<-Re(fft(c(0,neu2,0,Conj(neu2[N:1])),inverse=TRUE))/sqrt(n)
  }
  return(erg)
}
