##
# ------------------------------------------------------------------------
#
# "f_PseudoBlocks (x, s, eps_opt, delta_opt, p_XiXip1, func=sum)" --
#
# Compute the value of the function on pseudo-regenerative blocks.
#
# ------------------------------------------------------------------------
##
#' @aliases f_PseudoBlocks
#' @title Compute the Value of the Function on a (Pseudo)-Regenerative Blocks.
#' @description Function is an adaptation of \code{\link{GetPseudoBlocks}} 
#' to compute the value of any function on pseudo-regenerative blocks.
#' @param x  A vector or time series.
#' @param s A real number specifying the center of the small set. 
#' @param eps_opt A numeric value for the size of the small set.
#' @param delta_opt A numeric value for the lower bound in the minorization condition.
#' @param p_XiXip1 A numeric value representing the estimator of the transition density.
#' @param func A function to apply to each block. Default is \code{sum}.
#' @details This function computes the value of a specified function on pseudo-regenerative blocks 
#' of a time series. It uses parameters such as the central value (\code{s}), 
#' the size of the small set (\code{eps_opt}), 
#' and the lower bound in the minorization condition (\code{delta_opt}). 
#' Robustification is not proposed here 
#' due to the complexity of the pseudo-regenerative procedure.
#' @return A matrix with two columns: 
#' * \code{Subf} - the value of the function \code{func} on each sub-block,
#' * \code{lB} - the length of the sub-blocks on which the function is computed.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted. 
#' 
#'  Bertail, P. and Clémençon, S. (2006). Regenerative block bootstrap for Markov
#'  chains. \emph{Bernoulli}, \bold{12}, 689-712. 
#' @seealso \code{\link{GetPseudoBlocks}}, \code{\link{regenboot}}, \code{\link{findBestEpsilon}},
#' \code{\link{GetBlocks}}, \code{\link{smallEnsemble}}.
#' @keywords "Regenerative Block Bootstrap" "Pseudo-regenerative blocks" "Optimal small set"
#' @export
#' @examples
#' \donttest{
#' n=1000 
#' coeff=0.75
#' X = arima.sim(n=n, list(ar = c(coeff)))
#' sm <- findBestEpsilon(X,s=0,plotIt=FALSE)
#' eps = sm$epsilon
#' delta = sm$delta
#' m = sm$s
#' f = sm$trans
#' result <- f_PseudoBlocks(X, 0, eps, delta, f, func = max)
#' print(result)
#' }



f_PseudoBlocks <- function(x, s, eps_opt, delta_opt, p_XiXip1,func=sum) {
  # Check if 'x' is a numeric vector
  
  # Check if 's', 'eps_opt', 'delta_opt', and 'p_XiXip1' are numeric and of length 1
  if (!is.numeric(s) || length(s) != 1) {
    stop("Error: 's' must be a single numeric value.")
  }
  
  if (!is.numeric(eps_opt) || length(eps_opt) != 1) {
    stop("Error: 'eps_opt' must be a single numeric value.")
  }
  
  if (!is.numeric(delta_opt) || length(delta_opt) != 1) {
    stop("Error: 'delta_opt' must be a single numeric value.")
  }
  
  if (!is.numeric(p_XiXip1)) {
    stop("Error: 'p_XiXip1' must be a single numeric value.")
  }
  
  # Proceed with the rest of the function after passing all checks
  n = NROW(x) - 1
  pregen_opt = delta_opt * (x[1:n] >= s - eps_opt) * (x[1:n] <= s + eps_opt) *
    (x[2:(n + 1)] >= s - eps_opt) * (x[2:(n + 1)] <= s + eps_opt) / p_XiXip1
  regen = c((pregen_opt > runif(n)), 0)
  Bnumber = cumsum(c(0, regen[1:n]))
  
  if (regen[n] == 1) {
    NN = max(Bnumber)
  } else {
    NN = max(Bnumber) - 1
  }
  
  l_n = max(Bnumber)
  dataset = cbind(1:(n + 1), x, Bnumber, regen)
  
  Subf = matrix(0, NN, 1)
  lB = matrix(0, NN, 1)
  for (i in 1:NN) {
    aux = matrix(dataset[dataset[, 3] == i, 1:2], ncol = 2)
    Subf[i] = func(aux[, 2])
    lB[i] = nrow(aux)
  }
  Block = cbind(Subf,lB)
  return(Block)
}
