
#' Calculate the b-measure
#'
#' Function to calculate the b-measure, which quantifies the sensory 
#' differentiation retained. 
#'
#' @param Xb Three-way (\eqn{I \times J(J-1)/2 \times M}) array with 
#' \eqn{I} assessors, \eqn{J(J-1)/2} product comparisons, \eqn{M} CATA 
#' attributes, where values are counts of types \code{b} and \code{c} from the function 
#' \code{\link[cata]{barray}}.
#' @param Xc Array of same dimension as \code{Xb}, where values are counts of 
#' types \code{b} and \code{c} from the function \code{\link[cata]{barray}}.
#' @param oneI Indicates whether calculation is for one assessor (default: \code{FALSE}).
#' @param oneM Indicates whether calculation is for one attribute (default: \code{FALSE}).
#' @return b-measure
#' @author J.C. Castura
#' @references Castura, J.C., Meyners, M., Varela, P., & Næs, T. (2022). 
#' Clustering consumers based on product discrimination in check-all-that-apply 
#' (CATA) data. \emph{Food Quality and Preference}, 104564. 
#' \doi{10.1016/j.foodqual.2022.104564}.
#' @export
#' @examples
#' bread.bc <- barray(bread$cata[1:8,,1:5])
#' getb(bread.bc[,,1,], bread.bc[,,2,])
getb <- function(Xb, Xc, oneI = FALSE, oneM = FALSE){
  if(any(oneI, oneM)) {
    if(all(oneI, oneM)){
      dims <- c(1, length(Xb), 1)
    } else {
      dims <- dim(Xb) # 2D
      if(oneI){
        dims <- c(1, dims) # nI was 1
      }
      if(oneM){
        dims <- c(dims, 1) # nM was 1
      }
    }
    # make 3-way array
    Xb <- array(Xb, dims)
    Xc <- array(Xc, dims)
  }
  dims <- dim(Xb) # dimension is now nI x nJJ x nM
  if(!all.equal(dims, dim(Xc))){
    # dimension should also be nI x nJJ x nM
    return("Cannot calculate b-measure for unequal sized arrays")
  }
  # keep 3-way array structure
  return(sum(((apply(array(Xb - Xc, dims), 2:3, sum)^2) /
                (apply(array(Xb + Xc, dims), 2:3, sum))), na.rm = TRUE))
  
}

