% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sensitivity.R
\name{sensitivity}
\alias{sensitivity}
\title{Sensitivity Analysis Using R-Squared Values for Causal Decomposition Analysis}
\usage{
sensitivity(boot.res, fit.y, fit.m = NULL, mediator = NULL, covariates, group,
            sel.lev.group, conf.level = 0.95, max.rsq = 0.3)
}
\arguments{
\item{boot.res}{bootstrap results from an object fitted by the 'smi' function.}

\item{fit.y}{outcome model used in fitting the 'smi' function.
Can be of class 'lm' or 'glm'.}

\item{fit.m}{mediator model used in fitting the 'smi' function.
Can be of class 'lm', 'glm', or 'polr'.}

\item{mediator}{a vector containing the name of mediator used in the models.}

\item{covariates}{a vector containing the name of the covariate variable(s)
used in the models. Each covariate can be categorical with two or more
categories (two- or multi-valued factor) or continuous (numeric).}

\item{group}{a character string indicating the name of the social group indicator
such as race or gender (treatment) used in the models. The social group indicator
can be categorical with two or more categories (two- or multi-valued factor).}

\item{sel.lev.group}{a level of categorical social group indicator (treatment) which is to be
compared with the reference level.}

\item{conf.level}{level of the returned two-sided confidence intervals,
which are estimated by the nonparametric percentile bootstrap method.
Default is .95, which returns the 2.5 and 97.5 percentiles of the simulated
quantities.}

\item{max.rsq}{upper limit of the two sensitivity parameters (R-squared values).
Once it is set, the R-squared values between 0 and this upper limit are
explored to draw the sensitivity contour plots. Default is 0.3.}
}
\value{
\item{call}{original function call.}
\item{disparity.reduction}{a matrix containing the estimated disparity reduction
value along with lower and upper limits of the confidence interval, for each
combination of the two sensitivity parameters, assuming those two are equal.}
\item{disparity.remaining}{a matrix containing the estimated disparity remaining
value along with lower and upper limits of the confidence interval, for each
combination of the two sensitivity parameters, assuming those two are equal.}
\item{rm}{R-squared values between the mediator and unobserved confounder
(first sensitivity parameter), which are explored for the sensitivity analysis.}
\item{ry}{R-squared values between the outcome and unobserved confounder,
(second sensitivity parameter), which are explored for the sensitivity analysis.}
\item{red}{a matrix containing the estimated disparity reduction values given
each combination of the two sensitivity parameters.}
\item{lower_red}{a matrix containing the lower limit of disparity reduction given
each combination of the two sensitivity parameters.}
\item{rem}{a matrix containing the estimated disparity remaining values given
each combination of the two sensitivity parameters.}
\item{lower_rem}{a matrix containing the lower limit of disparity remaining given
each combination of the two sensitivity parameters.}
\item{RV_red}{robustness value for disparity reduction, which represents the
strength of association that will explain away the estimated disparity reduction.}
\item{RV_red_alpha}{robustness value for disparity reduction, which represents
the strength of association that will change the significance of the estimated
disparity reduction at the given significance level, assuming an equal association
to the mediator and the outcome.}
\item{RV_rem}{robustness value for disparity remaining, which represents the
strength of association that will explain away the estimated disparity remaining.}
\item{RV_rem_alpha}{robustness value for disparity remaining, which represents
the strength of association that will change the significance of the estimated
disparity remaining at the given significance level, assuming an equal association
to the mediator and the outcome.}
}
\description{
The function 'sensitivity' is used to implement sensitivity analysis for the
causal decomposition analysis, using R-squared values as sensitivity parameters.
}
\details{
This function is used to implement sensitivity analysis for the
causal decomposition analysis, using two sensitivity parameters: (i) the
R-squared value between the outcome and unobserved confounder given the social group indicator (treatment),
intermediate confounder, mediator, and covariates; and (ii) the R-squared value
between the mediator and unobserved confounder given the social group indicator (treatment), intermediate
confounder, and covariates (Park et al., 2023).

As of version 0.1.0, 'boot.res' must be fitted by the 'smi' function with a single
mediator, which can be of class 'lm', 'glm', or 'polr'.
}
\examples{
data(sMIDUS)

# Center covariates
sMIDUS$age <- scale(sMIDUS$age, center = TRUE, scale = FALSE)
sMIDUS$stroke <- scale(sMIDUS$stroke, center = TRUE, scale = FALSE)
sMIDUS$T2DM <- scale(sMIDUS$T2DM, center = TRUE, scale = FALSE)
sMIDUS$heart <- scale(sMIDUS$heart, center = TRUE, scale = FALSE)

fit.m <- lm(edu ~ racesex + age + stroke + T2DM + heart, data = sMIDUS)
fit.y <- lm(health ~ racesex + lowchildSES + abuse + edu + racesex:edu +
            age + stroke + T2DM + heart, data = sMIDUS)
smiRes <- smi(fit.m = fit.m, fit.y = fit.y, sims = 40, conf.level = .95,
          covariates = c("age", "stroke", "T2DM", "heart"), group = "racesex", seed = 227)
sensRes <- sensitivity(boot.res = smiRes, fit.m = fit.m, fit.y = fit.y, mediator = "edu",
                       covariates = c("age", "stroke", "T2DM", "heart"), group = "racesex",
                       sel.lev.group = "4", max.rsq = 0.3)

sensRes
names(sensRes) 
# Create sensitivity contour plots
plot(sensRes)
}
\references{
Park, S., Kang, S., Lee, C., & Ma, S. (2023). Sensitivity analysis for causal
decomposition analysis: Assessing robustness toward omitted variable bias,
Journal of Causal Inference, 11(1), 20220031.
}
\seealso{
\code{\link{smi}}
}
\author{
Suyeon Kang, University of Central Florida, \email{suyeon.kang@ucf.edu};
Soojin Park, University of California, Riverside, \email{soojinp@ucr.edu}.
}
