% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_dynamics.R
\name{simulate_dynamics}
\alias{simulate_dynamics}
\title{Simulate network state dynamics via SDEs (nonlinear, linear, or custom)}
\usage{
simulate_dynamics(
  adj_matrix,
  params,
  t_max = 100,
  dt = 0.1,
  S0 = NULL,
  model_type = "nonlinear",
  model_fn = NULL,
  stress_event = NULL,
  boundary = c("auto", "reflect", "clamp", "none"),
  clamp = NULL
)
}
\arguments{
\item{adj_matrix}{Numeric matrix (square; directed adjacency). Interpreted as i -> j.}

\item{params}{Named list of model parameters.
For \code{model_type = "nonlinear"}, requires vectors (length = n nodes):
\itemize{
\item \code{beta}: baseline/exogenous drive per node.
\item \code{alpha_self}: self-activation per node.
\item \code{delta}: nonlinear amplification of incoming effects.
\item \code{sigma}: noise SD per node.
}
For \code{model_type = "linear"}, requires \code{beta}, \code{alpha_self}, \code{sigma}.
For a custom model, include whatever your \code{model_fn} expects.}

\item{t_max}{Total simulated time (must be > 0).}

\item{dt}{Time step (must be > 0). The output has \code{floor(t_max/dt) + 1} rows.}

\item{S0}{Optional numeric vector of initial states (length = n). Defaults to 0.01.}

\item{model_type}{One of \code{"nonlinear"} (default), \code{"linear"}, or \code{NULL}
when using a custom \code{model_fn}.}

\item{model_fn}{Optional function with signature
\code{function(current, interaction, dt, ...)} returning a numeric
vector of increments \code{dS}. Additional args are taken from \code{params}.}

\item{stress_event}{Optional function \code{f(time, state) -> numeric(n)} that returns
an exogenous input vector added each step (e.g., shocks/perturbations).}

\item{boundary}{One of \code{"auto"}, \code{"reflect"}, \code{"clamp"}, \code{"none"}.
\itemize{
\item \code{"reflect"}: mirror overshoot back into \code{[clamp[1], clamp[2]]}.
\item \code{"clamp"}: hard-box to \code{[clamp[1], clamp[2]]}.
\item \code{"none"}: no bounding.
\item \code{"auto"}: pick a sensible default based on the model and \code{clamp}:
nonlinear -> \code{boundary = "reflect"} (and if \code{clamp} is \code{NULL}, use \code{c(0, 1)});
linear/custom -> \code{boundary = "none"} unless a clamp range is supplied, in which case use \code{"clamp"}.
}}

\item{clamp}{Either \code{NULL} (no numeric range) or a length-2 numeric vector
\code{c(min, max)} used by \code{"reflect"} or \code{"clamp"} to keep states within bounds.}
}
\value{
Numeric matrix of states over time (rows = time steps, cols = nodes).
The time vector is attached as \code{attr(result, "time")}.
}
\description{
Simulates the evolution of node states in a directed network using an
Euler–Maruyama discretization of stochastic differential equations (SDEs).
Choose the built-in nonlinear model, a linear alternative, or provide a
custom update function.
}
\details{
\strong{Direction convention.} By default \code{adj[i, j] = 1} encodes a directed edge
\code{i -> j}. Under this convention, the \emph{incoming input} to node \code{j} is the
dot product of column \code{j} with the current state; in vector form
\code{t(adj) \%*\% state}. If your internal convention differs, transpose accordingly.

Integration uses Euler–Maruyama. The per-step diffusion term is added
as \eqn{\sigma \sqrt{dt}\,Z} with \eqn{Z \sim \mathcal{N}(0, I)} (component-wise),
i.e., \code{sigma * sqrt(dt) * rnorm(n)}.
}
\section{Boundary handling}{

\itemize{
\item \strong{Reflecting} avoids “sticky” edges by bouncing trajectories back inside the range,
which is useful for bounded variables on \verb{[0,1]}.
\item \strong{Clamping} is numerically simple but can create artificial absorbing states at the limits.
\item For smoothly bounded dynamics, consider modeling on an unbounded latent scale and applying
a link (e.g., logistic) instead of hard post-step bounds.
}
}

\examples{
set.seed(1)
net <- matrix(c(0,1,0,0,
                0,0,1,0,
                0,0,0,1,
                1,0,0,0), 4, byrow = TRUE)

# Linear model, automatic boundary selection ("none" because no clamp supplied)
p_lin <- list(beta = rep(0.8, 4), alpha_self = rep(0.2, 4), sigma = rep(0.05, 4))
S1 <- simulate_dynamics(net, p_lin, model_type = "linear", boundary = "auto", t_max = 5, dt = 0.01)

# Linear model with a finite box -> "auto" switches to clamp on [0, 5]
S2 <- simulate_dynamics(net, p_lin, model_type = "linear",
                        boundary = "auto", clamp = c(0, 5), t_max = 5, dt = 0.01)

# Nonlinear model -> "auto" uses reflecting boundaries on [0,1]
p_nl <- list(beta = rep(0.2, 4), alpha_self = rep(0.2, 4),
             delta = rep(0.5, 4), sigma = rep(0.05, 4))
S3 <- simulate_dynamics(
  net, p_nl, model_type = "nonlinear",
  boundary = "auto", t_max = 5, dt = 0.01
)

}
