#' Scalar and atomic CLI assertions
#'
#' These functions wrap scalar and atomic checks from \pkg{checkmate} and either
#' return the underlying check result (`check_*_cli()`) or raise a
#' `cli::cli_abort()` error via [make_assertion()] (`assert_*_cli()`).
#' They are intended for validating basic building blocks of user input
#' (numbers, flags, strings, etc.) in a consistent way, while preserving the
#' semantics and arguments of the underlying \pkg{checkmate} checks.
#'
#' @param x Object to be checked.
#'   This is the value whose type, length, or other properties are validated.
#'
#' @param ... Additional arguments passed on to the corresponding
#'   \pkg{checkmate} function (e.g. `lower`, `upper`, `any.missing`,
#'   `all.missing`, `min.len`, `null.ok`).
#'
#' @param .var.name Character scalar used in error messages to refer to
#'   the checked object. Defaults to [checkmate::vname()], which tries to
#'   infer the variable name from the calling context.
#'
#' @param add Optional [checkmate::AssertCollection] to which
#'   assertion failures are added instead of triggering an immediate
#'   error. Defaults to `NULL`, which causes a `cli::cli_abort()` on
#'   failure.
#'
#' - Generic atomic containers:
#'   - [check_atomic_cli()], [assert_atomic_cli()] for atomic vectors of any
#'     storage mode.
#'   - [check_atomic_vector_cli()], [assert_atomic_vector_cli()] for atomic
#'     vectors with additional constraints (e.g., length, missingness).
#'
#' - Generic scalars:
#'   - [check_scalar_cli()], [assert_scalar_cli()] for length-one atomic
#'     values.
#'   - [check_scalar_na_cli()], [assert_scalar_na_cli()] for a single `NA`
#'     value.
#'
#' - Integer and numeric:
#'   - [check_integer_cli()], [assert_integer_cli()] for integer vectors.
#'   - [check_integerish_cli()], [assert_integerish_cli()] for integer-like
#'     numerics (e.g., `1`, `2.0`).
#'   - [check_int_cli()], [assert_int_cli()] for integer vectors (thin wrapper
#'     around [checkmate::check_int()]).
#'   - [check_numeric_cli()], [assert_numeric_cli()] for numeric vectors of any
#'     type.
#'   - [check_number_cli()], [assert_number_cli()] for numeric scalars
#'     (length-one).
#'   - [check_double_cli()], [assert_double_cli()] for double vectors.
#'   - [check_count_cli()], [assert_count_cli()] for non-negative integer
#'     counts.
#'   - [check_complex_cli()], [assert_complex_cli()] for complex vectors.
#'
#' - Logical and character:
#'   - [check_flag_cli()], [assert_flag_cli()] for single logical flags.
#'   - [check_logical_cli()], [assert_logical_cli()] for logical vectors.
#'   - [check_true_cli()], [assert_true_cli()] for conditions that must be
#'     `TRUE`.
#'   - [check_false_cli()], [assert_false_cli()] for conditions that must be
#'     `FALSE`.
#'   - [check_string_cli()], [assert_string_cli()] for length-one character
#'     strings.
#'   - [check_character_cli()], [assert_character_cli()] for character
#'     vectors.
#'
#' - Special values:
#'   - [check_null_cli()], [assert_null_cli()] for `NULL` values.
#'
#' In all `assert_*_cli()` functions, `x` is the object being checked and
#' `.var.name` is used only for error message construction; the return value is
#' always `x` (invisibly) on success.
#'
#' @returns
#'
#' - `check_*_cli()` functions return `TRUE` on success or a character vector
#'   describing the failure, exactly like the corresponding
#'   \pkg{checkmate} checks.
#' - `assert_*_cli()` functions return `x` invisibly on success and either:
#'   - raise a `cli::cli_abort()` error with bullet-style messages, or
#'   - push messages into an [checkmate::AssertCollection] if `add` is supplied.
#'
#' @examples
#'
#' # Generic atomic and scalar:
#'
#' # Atomic vs list
#' check_atomic_cli(1:3)
#' try(check_atomic_cli(list(1, 2)))                   # failure: not atomic
#'
#' assert_atomic_cli(1:3)
#' try(assert_atomic_cli(list(1, 2)))
#'
#' # Atomic vector vs scalar
#' check_atomic_vector_cli(1:3)
#' try(check_atomic_vector_cli(matrix(1:4, 2)))        # failure
#'
#' assert_atomic_vector_cli(1:3)
#'
#' # Scalars
#' check_scalar_cli(1L)
#' try(check_scalar_cli(1:3))                          # failure: length > 1
#'
#' assert_scalar_cli("id")
#' try(assert_scalar_cli(c("a", "b")))
#'
#' check_scalar_na_cli(NA)
#' try(check_scalar_na_cli(c(NA, NA)))                 # failure
#'
#' assert_scalar_na_cli(NA)
#'
#' # Integer and numeric:
#'
#' # Integer vectors
#' check_integer_cli(1:5)
#' try(check_integer_cli(c(1, 2, 3.5)))                # failure
#'
#' assert_integer_cli(1:3)
#' try(assert_integer_cli(c(1, 2, 2.5)))
#'
#' # Integerish (numeric but whole)
#' check_integerish_cli(c(1, 2, 3))
#' try(check_integerish_cli(c(1, 2.5)))                # failure
#'
#' assert_integerish_cli(c(1, 2, 3))
#'
#' # `check_int` is a thin wrapper for integer vectors
#' check_int_cli(1L)
#' try(check_int_cli(1.5))                             # failure
#'
#' assert_int_cli(1L)
#' try(assert_int_cli(1.5))
#'
#' # Numeric vs number
#' check_numeric_cli(c(0.1, 0.2))
#' try(check_numeric_cli("a"))                         # failure
#'
#' assert_numeric_cli(c(1, 2, 3))
#'
#' check_number_cli(3.14)
#' try(check_number_cli(c(1, 2)))                      # failure: not scalar
#'
#' assert_number_cli(0)
#' try(assert_number_cli(c(0, 1)))
#'
#' # Double and count
#' check_double_cli(c(0.1, 0.2))
#' try(check_double_cli("a"))                          # failure
#'
#' count_vals <- c(0L, 10L)
#' check_count_cli(0L)
#' try(check_count_cli(-1L))                           # failure: negative
#'
#' assert_count_cli(10L)
#' try(assert_count_cli(-5L))
#'
#' # Complex, character, and logical:
#'
#' # Complex
#' check_complex_cli(1 + 2i)
#' try(check_complex_cli(1:3))                         # failure
#'
#' assert_complex_cli(1 + 2i)
#' try(assert_complex_cli(1:3))
#'
#' # String vs character vector
#' check_string_cli("species_id")
#' try(check_string_cli(c("a", "b")))                  # failure: length > 1
#'
#' assert_string_cli("ok")
#' try(assert_string_cli(c("a", "b")))
#'
#' check_character_cli(c("a", "b"))
#' try(check_character_cli(1:3))                       # failure
#'
#' assert_character_cli(c("a", "b"))
#'
#' # Logical flags, TRUE/FALSE, and NULL:
#'
#' # Single logical flag
#' check_flag_cli(TRUE)
#' try(check_flag_cli(c(TRUE, FALSE)))                 # failure
#'
#' assert_flag_cli(FALSE)
#' try(assert_flag_cli(c(TRUE, FALSE)))
#'
#' # Logical vectors
#' check_logical_cli(c(TRUE, FALSE, TRUE))
#' try(check_logical_cli(c(1, 0, 1)))                  # failure
#'
#' assert_logical_cli(c(TRUE, FALSE))
#'
#' # Conditions that must be TRUE / FALSE
#' check_true_cli(1 < 2)
#' try(check_true_cli(FALSE))                          # failure
#'
#' assert_true_cli(1 == 1)
#' try(assert_true_cli(1 == 2))
#'
#' check_false_cli(FALSE)
#' try(check_false_cli(TRUE))                          # failure
#'
#' assert_false_cli(1 > 2)
#' try(assert_false_cli(1 < 2))
#'
#' # NULL checks
#' check_null_cli(NULL)
#' try(check_null_cli(1))                              # failure
#'
#' assert_null_cli(NULL)
#' try(assert_null_cli("not null"))
#'
#' @seealso
#'   [checkmate::check_atomic()], [checkmate::check_scalar()],
#'   [checkmate::check_integer()], [checkmate::check_integerish()],
#'   [checkmate::check_int()], [checkmate::check_numeric()],
#'   [checkmate::check_number()], [checkmate::check_double()],
#'   [checkmate::check_complex()], [checkmate::check_count()],
#'   [checkmate::check_string()], [checkmate::check_flag()],
#'   [checkmate::check_logical()], [checkmate::check_character()],
#'   [checkmate::check_null()], [checkmate::check_true()],
#'   [checkmate::check_false()], [make_assertion()], [assert_cli()]
#'
#' @name checkcli-scalars
#' @family checkCLI
#' @import checkmate
#' @import cli
#' @importFrom glue glue
#' @importFrom purrr map2

#' @rdname checkcli-scalars
#' @export
check_atomic_cli <- function(...) {
  checkmate::check_atomic(...)
}

#' @rdname checkcli-scalars
#' @export
assert_atomic_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_atomic(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_atomic_vector_cli <- function(...) {
  checkmate::check_atomic_vector(...)
}

#' @rdname checkcli-scalars
#' @export
assert_atomic_vector_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_atomic_vector(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_scalar_cli <- function(...) {
  checkmate::check_scalar(...)
}

#' @rdname checkcli-scalars
#' @export
assert_scalar_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_scalar(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_scalar_na_cli <- function(...) {
  checkmate::check_scalar_na(...)
}

#' @rdname checkcli-scalars
#' @export
assert_scalar_na_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_scalar_na(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_integer_cli <- function(...) {
  checkmate::check_integer(...)
}

#' @rdname checkcli-scalars
#' @export
assert_integer_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_integer(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_integerish_cli <- function(...) {
  checkmate::check_integerish(...)
}

#' @rdname checkcli-scalars
#' @export
assert_integerish_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_integerish(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_double_cli <- function(...) {
  checkmate::check_double(...)
}

#' @rdname checkcli-scalars
#' @export
assert_double_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_double(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_complex_cli <- function(...) {
  checkmate::check_complex(...)
}

#' @rdname checkcli-scalars
#' @export
assert_complex_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_complex(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_count_cli <- function(...) {
  checkmate::check_count(...)
}

#' @rdname checkcli-scalars
#' @export
assert_count_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_count(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_string_cli <- function(...) {
  checkmate::check_string(...)
}

#' @rdname checkcli-scalars
#' @export
assert_string_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_string(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_flag_cli <- function(...) {
  checkmate::check_flag(...)
}

#' @rdname checkcli-scalars
#' @export
assert_flag_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_flag(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_int_cli <- function(...) {
  checkmate::check_int(...)
}

#' @rdname checkcli-scalars
#' @export
assert_int_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_int(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_numeric_cli <- function(...) {
  checkmate::check_numeric(...)
}

#' @rdname checkcli-scalars
#' @export
assert_numeric_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_numeric(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_number_cli <- function(...) {
  checkmate::check_number(...)
}

#' @rdname checkcli-scalars
#' @export
assert_number_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_number(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_logical_cli <- function(...) {
  checkmate::check_logical(...)
}

#' @rdname checkcli-scalars
#' @export
assert_logical_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_logical(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_character_cli <- function(...) {
  checkmate::check_character(...)
}

#' @rdname checkcli-scalars
#' @export
assert_character_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_character(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_null_cli <- function(...) {
  checkmate::check_null(...)
}

#' @rdname checkcli-scalars
#' @export
assert_null_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_null(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_true_cli <- function(...) {
  checkmate::check_true(...)
}

#' @rdname checkcli-scalars
#' @export
assert_true_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_true(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-scalars
#' @export
check_false_cli <- function(...) {
  checkmate::check_false(...)
}

#' @rdname checkcli-scalars
#' @export
assert_false_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_false(x, ...),
      .var.name,
      add
    )
  )
}
