% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cvforecast.R
\name{cvforecast}
\alias{cvforecast}
\alias{print.cvforecast}
\alias{summary.cvforecast}
\alias{print.summary.cvforecast}
\title{Time series cross-validation forecasting}
\usage{
cvforecast(
  y,
  forecastfun,
  h = 1,
  level = c(80, 95),
  forward = TRUE,
  xreg = NULL,
  initial = 1,
  window = NULL,
  ...
)
}
\arguments{
\item{y}{Univariate time series.}

\item{forecastfun}{Function to return an object of class \code{"forecast"}.
Its first argument must be a univariate time series, and it must have an
argument \code{h} for the forecast horizon and an argument \code{level} for
the confidence level for prediction intervals. If exogenous predictors are used,
then it must also have \code{xreg} and \code{newxreg} arguments corresponding
to the training and test periods, respectively.}

\item{h}{Forecast horizon.}

\item{level}{Confidence level for prediction intervals.
If \code{NULL}, prediction intervals will not be generated.}

\item{forward}{If \code{TRUE}, the final forecast origin for forecasting is
\eqn{y_T}. Otherwise, the final forecast origin is \eqn{y_{T-1}}.}

\item{xreg}{Exogenous predictor variables passed to \code{forecastfun} if required.
It should be of the same size as \code{y}+\code{forward}*\code{h}, otherwise,
\code{NA} padding or subsetting will be applied.}

\item{initial}{Initial period of the time series where no cross-validation
forecasting is performed.}

\item{window}{Length of the rolling window. If \code{NULL}, a rolling window
will not be used.}

\item{...}{Other arguments are passed to \code{forecastfun}.}
}
\value{
A list of class \code{c("cvforecast", "forecast")} with components:
\item{x}{The original time series.}
\item{series}{The name of the series \code{x}.}
\item{xreg}{Exogenous predictor variables used in the model, if applicable.}
\item{method}{A character string "cvforecast".}
\item{fit_times}{The number of times the model is fitted in cross-validation.}
\item{MEAN}{Point forecasts as a multivariate time series, where the \eqn{h}th column
holds the point forecasts for forecast horizon \eqn{h}. The time index
corresponds to the period for which the forecast is produced.}
\item{ERROR}{Forecast errors given by
\eqn{e_{t+h|t} = y_{t+h}-\hat{y}_{t+h|t}}{e[t+h] = y[t+h]-f[t+h]}.}
\item{LOWER}{A list containing lower bounds for prediction intervals for
each \code{level}. Each element within the list will be a multivariate time
series with the same dimensional characteristics as \code{MEAN}.}
\item{UPPER}{A list containing upper bounds for prediction intervals for
each \code{level}. Each element within the list will be a multivariate time
series with the same dimensional characteristics as \code{MEAN}.}
\item{level}{The confidence values associated with the prediction intervals.}
\item{call}{The matched call.}
\item{forward}{Whether \code{forward} is applied.}
If \code{forward} is \code{TRUE}, the components \code{mean}, \code{lower},
\code{upper}, and \code{model} will also be returned, showing the information
about the final fitted model and forecasts using all available observations, see
e.g. \code{\link[forecast]{forecast.ets}} for more details.
}
\description{
Compute forecasts and other information by applying
\code{forecastfun} to subsets of the time series \code{y} using a
rolling forecast origin.
}
\details{
Let \code{y} denote the time series \eqn{y_1,\dots,y_T}{y[1:T]} and
let \eqn{t_0} denote the initial period.

Suppose \code{forward = TRUE}. If \code{window} is \code{NULL},
\code{forecastfun} is applied successively to the subset time series
\eqn{y_{1},\dots,y_t}{y[1:t]}, for \eqn{t=t_0,\dots,T},
generating forecasts \eqn{\hat{y}_{t+1|t},\dots,\hat{y}_{t+h|t}}{f[t+1:h]}. If \code{window} is not
\code{NULL} and has a length of \eqn{t_w}, then \code{forecastfun} is applied
successively to the subset time series \eqn{y_{t-t_w+1},\dots,y_{t}}{y[(t-t_w+1):t)]},
for \eqn{t=\max(t_0, t_w),\dots,T}.

If \code{forward} is \code{FALSE}, the last observation used for training will
be \eqn{y_{T-1}}.
}
\examples{
# Simulate time series from an AR(2) model
library(forecast)
series <- arima.sim(n = 200, list(ar = c(0.8, -0.5)), sd = sqrt(1))

# Example with a rolling window
far2 <- function(x, h, level) {
  Arima(x, order = c(2, 0, 0)) |>
    forecast(h = h, level)
}
fc <- cvforecast(series, forecastfun = far2, h = 3, level = 95,
                 forward = TRUE, initial = 1, window = 50)
print(fc)
summary(fc)

# Example with exogenous predictors
far2_xreg <- function(x, h, level, xreg, newxreg) {
  Arima(x, order=c(2, 0, 0), xreg = xreg) |>
    forecast(h = h, level = level, xreg = newxreg)
}
fc_xreg <- cvforecast(series, forecastfun = far2_xreg, h = 3, level = 95,
                      forward = TRUE, xreg = matrix(rnorm(406), ncol = 2, nrow = 203),
                      initial = 1, window = 50)

}
