% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main_wrapper.R
\name{run_complete_factor_analysis_robust}
\alias{run_complete_factor_analysis_robust}
\title{Complete factor-OU convergence analysis pipeline}
\usage{
run_complete_factor_analysis_robust(
  X_matrix,
  Y_matrix,
  TMG = NULL,
  COM_matrix = NULL,
  SPVR_matrix = NULL,
  CA = NULL,
  sector_names = NULL,
  max_comp = 3,
  dfm_lags = 1,
  ou_chains = 10,
  ou_iter = 10000,
  skip_ou = FALSE,
  run_convergence_tests = TRUE,
  path_cpi = NULL,
  path_weights = NULL,
  verbose = TRUE
)
}
\arguments{
\item{X_matrix}{Matrix of first set of variables}

\item{Y_matrix}{Matrix of second set of variables}

\item{TMG}{Optional TMG matrix (default: NULL)}

\item{COM_matrix}{Optional COM matrix (default: NULL)}

\item{SPVR_matrix}{Optional SPVR matrix (default: NULL)}

\item{CA}{Optional CA parameter (default: NULL)}

\item{sector_names}{Vector of sector names (default: NULL)}

\item{max_comp}{Maximum number of components (default: 3)}

\item{dfm_lags}{Number of lags for DFM (default: 1)}

\item{ou_chains}{Number of MCMC chains for OU estimation (default: 10)}

\item{ou_iter}{Number of MCMC iterations (default: 10000)}

\item{skip_ou}{Logical, skip OU estimation (default: FALSE)}

\item{run_convergence_tests}{Logical, run convergence tests (default: TRUE)}

\item{path_cpi}{Path to CPI data (default: NULL)}

\item{path_weights}{Path to weights data (default: NULL)}

\item{verbose}{Logical; print progress and diagnostic information. Default \code{TRUE}.}
}
\value{
List with components:
\describe{
\item{\code{factors}}{List containing PLS-extracted factors (scores_X, scores_Y)
and related objects (pls_X, pls_Y, ncomp_X, ncomp_Y).}
\item{\code{dfm}}{List with DFM estimation results including VAR fit, lag order,
diagnostics, and optional impulse responses.}
\item{\code{factor_ou}}{List with Factor-OU model estimates: beta, lambda,
sigma, half_life, method used, and optional Stan fit object.}
\item{\code{convergence_tests}}{List with formal convergence test results
(if \code{run_convergence_tests = TRUE}).}
\item{\code{robustness_tests}}{List with robustness test results (if
\code{run_robustness_tests = TRUE}).}
\item{\code{diagnostics}}{List with data diagnostics (multicollinearity,
stationarity, structural breaks).}
\item{\code{bayesian_cpi}}{List with Bayesian disaggregation results (if
CPI paths provided).}
\item{\code{metadata}}{List with analysis metadata (timestamp, versions,
parameters).}
}
}
\description{
Executes the end-to-end analysis workflow: data preparation, PLS-based factor
extraction, DFM estimation, Factor-OU estimation, convergence tests, and
robustness checks. This is the main user-facing function.
}
\details{
uses column names of \code{X_matrix}.
and reweighting tests). Can be \code{NULL}.

This function orchestrates the complete analysis:
\enumerate{
\item Data validation and diagnostics
\item Bayesian CPI disaggregation (if applicable)
\item PLS-based factor extraction with optimal component selection
\item Dynamic Factor Model estimation via VAR
\item Factor Ornstein-Uhlenbeck mean-reversion model
\item Formal convergence tests (stationarity, cointegration, speed)
\item Robustness tests (permutation, reweighting, jackknife)
}
}
\examples{
\donttest{
# Basic usage with simulated data
set.seed(123)
X <- matrix(rnorm(100 * 10), 100, 10)
Y <- X + matrix(rnorm(100 * 10, 0, 0.5), 100, 10)

results <- run_complete_factor_analysis_robust(
  X_matrix = X,
  Y_matrix = Y,
  max_comp = 3,
  dfm_lags = 1,
  ou_chains = 4,
  ou_iter = 2000,
  verbose = FALSE
)

# View convergence summary
summary(results$convergence_tests)

# Visualize results
visualize_factor_dynamics(
  dfm_result = results$dfm,
  ou_result = results$factor_ou,
  factors_data = results$factors
)
}


}
\seealso{
\code{\link{estimate_DFM}}, \code{\link{estimate_factor_OU}},
\code{\link{run_convergence_robustness_tests}}, \code{\link{visualize_factor_dynamics}}
}
