% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gg_km.R
\name{gg_km}
\alias{gg_km}
\alias{process_survfit}
\title{Kaplan-Meier Plot}
\usage{
process_survfit(fit_km, strata_levels = "All", max_time = NULL)

gg_km(
  surv_plot_data,
  lty = NULL,
  lwd = 0.5,
  censor_show = TRUE,
  size = 2,
  max_time = NULL,
  xticks = NULL,
  yval = c("Survival", "Failure"),
  ylim = NULL,
  font_size = 10,
  legend_pos = NULL
)
}
\arguments{
\item{fit_km}{A fitted Kaplan-Meier object of class \code{survfit}.}

\item{strata_levels}{(\code{string})\cr
A single character string used as the strata level if the input \code{fit_km} object
has no strata (e.g., \code{"All"}).}

\item{max_time}{(\code{numeric})\cr
A single numeric value defining the \strong{maximum time point} to display on the x-axis.}

\item{surv_plot_data}{(\code{data.frame})\cr
A data frame containing the pre-processed survival data, ready for plotting.
This data should be equivalent to the output of \code{process_survfit}.}

\item{lty}{(\code{numeric} or \code{NULL})\cr
A numeric vector of \strong{line types} (e.g., \code{1} for solid, \code{2} for dashed) for the survival curves, or \code{NULL} for
\code{ggplot2} defaults. The length should match the number of arms/groups.}

\item{lwd}{(\code{numeric})\cr
A single numeric value specifying the \strong{line width} for the survival curves.}

\item{censor_show}{(\code{logical})\cr
A single logical value indicating whether to display \strong{censoring marks} on the plot. Defaults to \code{TRUE}.}

\item{size}{(\code{numeric})\cr
A single numeric value specifying the \strong{size} of the censoring marks.}

\item{xticks}{(\code{numeric} or \code{NULL})\cr
A numeric vector of explicit \strong{x-axis tick positions}, or a single numeric value representing the \strong{interval}
between ticks, or \code{NULL} for automatic \code{ggplot2} scaling.}

\item{yval}{(\code{character})\cr
A single character string, either \code{"Survival"} or \code{"Failure"} to plot the corresponding probability.}

\item{ylim}{(\code{numeric})\cr
A \strong{numeric vector of length 2} defining the lower and upper limits of the y-axis (e.g., \code{c(0, 1)}).}

\item{font_size}{(\code{numeric})\cr
A single numeric value specifying the \strong{base font size} for the plot theme elements.}

\item{legend_pos}{(\code{numeric} or \code{NULL})\cr
A \strong{numeric vector of length 2} defining the \strong{legend position} as (x, y) coordinates relative to the plot
area (ranging from 0 to 1), or \code{NULL} for automatic placement.}
}
\value{
The function \code{process_survfit} returns a data frame containing the survival
curve steps, confidence intervals, and censoring info.

The function \code{gg_km} returns a \code{ggplot2} object of the KM plot.
}
\description{
This set of functions facilitates the creation of Kaplan-Meier survival plots using \code{ggplot2}. Use
\code{process_survfit()} to prepare the survival data from a fitted \code{survfit} object, and then
\code{gg_km()} to generate the Kaplan-Meier plot with various customization options. Additional functions
like \code{annot_surv_med()}, \code{annot_cox_ph()}, and \code{annotate_riskdf()} allow for adding summary tables and
annotations to the plot.
}
\details{
Data setup assumes \code{"time"} is event time, \code{"status"} is event indicator (\code{1} represents an event),
while \code{"arm"} is the treatment group.
}
\section{Functions}{
\itemize{
\item \code{process_survfit()}: takes a fitted \link[survival:survfit]{survival::survfit} object and processes it into a data frame
suitable for plotting a Kaplan-Meier curve with \code{ggplot2}. Time zero is also added to the data.

\item \code{gg_km()}: creates a Kaplan-Meier survival curve, with
support for various customizations like censoring marks, Confidence  Intervals (CIs), and axis control.

}}
\examples{
# Data preparation for KM plot
use_lung <- survival::lung
use_lung$arm <- factor(sample(c("A", "B", "C"), nrow(use_lung), replace = TRUE))
use_lung$status <- use_lung$status - 1 # Convert status to 0/1
use_lung <- na.omit(use_lung)

# Fit Kaplan-Meier model
formula <- survival::Surv(time, status) ~ arm
fit_kmg01 <- survival::survfit(formula, use_lung)

# Process survfit data for plotting
surv_plot_data <- process_survfit(fit_kmg01)
head(surv_plot_data)

# Example of making the KM plot
plt_kmg01 <- gg_km(surv_plot_data)

# Confidence Interval as Ribbon
plt_kmg01 +
  ggplot2::geom_ribbon(alpha = 0.3, lty = 0, na.rm = TRUE)

# Adding Title and Footnotes
plt_kmg01 +
  ggplot2::labs(title = "title", caption = "footnotes")

# Changing xlab and ylab
plt_kmg01 +
  ggplot2::xlab("Another Day") +
  ggplot2::ylab("THE Survival Probability")

}
