#' Summarize DRI Permutation Test Results
#'
#' `summarize_perm_dri` summarizes the results of a permutation test done using
#' `permute_dri(..., summary = FALSE)`; useful for summarizing results after
#' plotting permutation results
#'
#' @param perms results of the permutation test generated by `permute_dri()`
#' @param type which type of statistics to summarize (e.g., `"common"`,
#' `"robust"`, `"mean"`)
#'
#' @importFrom rstatix get_summary_stats
#' @import dplyr
#' @importFrom rlang .data
#'
#' @returns summary of permutation test
#'
#' @seealso [permute_dri()] for generating the `perms` parameter
#' @seealso [rstatix::get_summary_stats()] for values of `type`
#' @export
#'
#' @examples
#'
#' # get pre-deliberation (stage_id == 1) data from Zukunft case
#' data <- human_data[human_data$stage_id == 1 & human_data$case == "Zukunft", ]
#'
#' # create permutations
#' perms <- permute_dri(data, iterations = 100, summary = FALSE)
#'
#' summarize_perm_dri(perms)
#'
summarize_perm_dri <- function(perms, type = "common") {

  # get observed dri
  obs_dri <- perms[perms$source == "observed",]$dri

  # get all permutations
  perm_dri <- perms[perms$source == "permutation",]

  # get permutation summary
  perm_summ <- perm_dri %>%
    mutate(perm_dri = .data$dri) %>%
    select(perm_dri) %>%
    get_summary_stats(type = type) %>%
    select(-.data$variable)

  # calculate p
  p <- nrow(perm_dri %>% filter(.data$dri >= obs_dri)) / nrow(perm_dri)

  # compile summary
  summ <- tibble(
    obs_dri,
    p,
    perm_summ
  )

  return(summ)

}
