% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/outliers_epoch.R
\name{outliers_epoch}
\alias{outliers_epoch}
\title{Select outlier epochs}
\usage{
outliers_epoch(
  data,
  amplitude = "signal",
  time = NULL,
  method = c("iqr", "percentile", "hampel"),
  k_iqr = 1.5,
  k_mad = 3,
  p = 0.975,
  print_tab = TRUE
)
}
\arguments{
\item{data}{A data frame, tibble or a database table with input data, required columns: \code{time}, \code{epoch} and the column with EEG amplitude specified by \code{amplitude} parameter. Optional columns: \code{group}, \code{subject}, \code{sensor}, \code{condition}.}

\item{amplitude}{A character specifying the name of the column from input data with an EEG amplitude values. Default is \code{"signal"}.}

\item{time}{A vector with time range for outliers detection. If not defined, the outliers are searched for all time points in the dataset.}

\item{method}{A character denoting the method used for outlier detection. The options are: \code{"iqr"} for interquartile range (IQR) criterion (default value), \code{"percentile"} for percentile method and \code{"hampel"} for Hampel filter method. See details for further information about methods.}

\item{k_iqr}{A positive numeric value denoting the scaling factor used in the IQR criterion. Default value is \code{k_iqr = 1.5}.}

\item{k_mad}{A positive numeric value denoting the scaling factor used in the Hampel filter method. Default value is \code{k_mad = 3}.}

\item{p}{A probability value from \code{[0,1]} interval determining percentile to the percentile method (according to \code{probs} argument in \code{quantile()} function). The default value is set to 0.975 for the interval formed by the 2.5 and 97.5 percentiles.}

\item{print_tab}{Logical. Indicates, whether result table should be printed in console. Default is \code{TRUE}.}
}
\value{
A list with following components:
\item{epoch_table}{A data frame with epoch ID and the number of time points in which the epoch was evaluated as outlier. (Only epochs with occurrence of outliers in at least one time point are included.)}
\item{outliers_data}{A data frame with subset of data corresponding to the outliers found. (The full record for each flagged point from \code{epoch_table}.)}
With the setting \code{print_tab = TRUE}, the \code{epoch_table} is also printed to the console.
}
\description{
Function identifies epochs with outlier values in a numeric EEG amplitude variable in chosen time points.
Outliers are detected separately at each time point within the groups present in the data. The function then summarizes how many times each epoch was marked as an outlier
across all time points.

Epochs are marked as outliers based on one of the following criteria: interquartile range criterion, percentile approach or Hampel filter method.
}
\details{
The input data frame or database table must contain at least following columns:
\code{epoch} - a column with epoch numbers/labels,
\code{time} - a column with time point numbers,
signal (or other name specified in \code{amplitude} parameter) - a column with measured EEG signal values.

The outlier detection method is chosen through \code{method} argument. The possibilities are
\itemize{
\item \code{iqr}: The interquartile range criterion, values outside the interval \code{[lower quartile - k_iqr * IQR, upper quartile + k_iqr * IQR]} are considered as outliers. IQR denotes interquartile range and \code{k_iqr} the scaling factor.
\item \code{percentile}: The percentile method, values outside the interval defined by the chosen percentiles are considered as outliers. Note: chosing small \code{p}leads to marking too many (or all) values.
\item \code{hampel}: The Hampel filter method, values outside the interval \code{[median - k_mad * MAD, median + k_mad * MAD]} are considered as outliers. MAD denotes median absolute deviation and \code{k_mad} the scaling factor.
Each of the above methods operates independently per time point, not globally across time.
}

Note: For large datasets, the calculation can be time-consuming. It is recommended to pre-filter or subset the data before using this function to reduce computation time.
}
\examples{
# 1. Outlier epoch detection for subject 2, electrode E45 for the whole time range with IQR method
epochdata |>
pick_data(subject_rg = 2, sensor_rg = "E45") |>
outliers_epoch(amplitude = "signal")
## From the result table we see that epochs 14 and 15 were marked as outliers in 50 cases out of 50

# 2. Outlier epoch detection for both subjects, electrode E45 for the whole time range
# using percentile method with 1 and 99 percentiles
outdata <- epochdata |>
pick_data(sensor_rg = "E45") |>
outliers_epoch(amplitude = "signal", method = "percentile", p = 0.99)
# see head of outliers data
head(outdata$outliers_data)
}
