% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/boot.matrix.R
\name{boot.matrix}
\alias{boot.matrix}
\title{Bootstraps and rarefies data.}
\usage{
boot.matrix(
  data,
  bootstraps = 100,
  boot.type = "full",
  boot.by = "rows",
  rarefaction = FALSE,
  verbose = FALSE,
  prob = NULL
)
}
\arguments{
\item{data}{A \code{matrix} or a list of matrices (typically output from \link{chrono.subsets} or \link{custom.subsets} - see details).}

\item{bootstraps}{The number of bootstrap pseudoreplicates (\code{default = 100}).}

\item{boot.type}{The bootstrap algorithm to use (\code{default = "full"}; see details).}

\item{boot.by}{Which dimension of the data to bootstrap: either \code{"rows"} to bootstrap the elements (default), \code{"columns"} for the dimensions or \code{"dist"} for bootstrapping both equally (e.g. for distance matrices).}

\item{rarefaction}{Either a \code{logical} value whether to fully rarefy the data, a set of \code{numeric} values used to rarefy the data or \code{"min"} to rarefy at the minimum level (see details).}

\item{verbose}{A \code{logical} value indicating whether to be verbose or not.}

\item{prob}{Optional, a \code{matrix} or a \code{vector} of probabilities for each element to be selected during the bootstrap procedure. The \code{matrix} or the \code{vector} must have a row names or names attribute that corresponds to the elements in \code{data}.}
}
\value{
This function outputs a \code{dispRity} object containing:
\item{matrix}{the multidimensional space (a \code{matrix}).}
\item{call}{A \code{list} containing the called arguments.}
\item{subsets}{A \code{list} containing matrices pointing to the elements present in each subsets.}

Use \link{summary.dispRity} to summarise the \code{dispRity} object.
}
\description{
Bootstraps and rarefies either a matrix or a list of matrices.
}
\details{
\code{data}: The data is considered as the multidimensional space and is not transformed (e.g. if ordinated with negative eigen values, no correction is applied to the matrix).

\code{rarefaction}: when the input is \code{numeric}, the number of elements is set to the value(s) for each bootstrap. If some subsets have fewer elements than the rarefaction value, the subsets is not rarefied.
When the input is \code{"min"}, the smallest number of elements is used (or 3 if some subsets have less than 3 elements).

\code{boot.type}: the different bootstrap algorithms are:
\itemize{
  \item \code{"full"}: resamples all the rows of the matrix and replaces them with a new random sample of rows (with \code{replace = TRUE}, meaning all the elements can be duplicated in each bootstrap).
  \item \code{"single"}: resamples only one row of the matrix and replaces it with a new randomly sampled row (with \code{replace = FALSE}, meaning that only one element can be duplicated in each bootstrap).
  \item \code{"null"}: resamples all rows of the matrix across subsets. I.e. for each subset of \emph{n} elements, this algorithm resamples \emph{n} elements across \emph{ALL} subsets. If only one subset (or none) is used, this does the same as the \code{"full"} algorithm.
}

\code{prob}: This option allows to attribute specific probability to each element to be drawn.
A probability of 0 will never sample the element, a probability of 1 will always allow it to be sampled.
This can also be useful for weighting elements: an element with a weight of 10 will be sampled ten times more.
If the argument is a \code{matrix}, it must have rownames attributes corresponding to the element names.
If the argument is a \code{vector}, it must have names attributes corresponding to the element names.

Multiple trees: If the given \code{data} is a \code{\link{chrono.subsets}} based on multiple trees, the sampling is proportional to the presence of each element in each tree: \eqn{\sum (1/n) / T} (with \emph{n} being the maximum number of elements among the trees and \emph{T} being the total numbers of trees).
For example, for a slice through two trees resulting in the selection of elements \code{A} and \code{B} in the first tree and \code{A}, \code{B} and \code{C} in the second tree, the \code{"full"} bootstrap algorithm will select three elements (with replacement) between \code{A}, \code{B} and \code{C} with a probability of respectively \eqn{p(A) = 1/3} (\eqn{p(A) = (1/3 + 1/3) / 2}), \eqn{p(B) = 1/3} and \eqn{p(C) = 1/6} (\eqn{p(C) = (0 + 1/3) / 2}).
}
\examples{
## Load the Beck & Lee 2014 matrix
data(BeckLee_mat50)

## Bootstrapping a matrix
## Bootstrapping an ordinated matrix 20 times
boot.matrix(BeckLee_mat50, bootstraps = 20)
## Bootstrapping an ordinated matrix with rarefaction
boot.matrix(BeckLee_mat50, bootstraps = 20, rarefaction = TRUE)
## Bootstrapping an ordinated matrix with only elements 7, 10 and 11 sampled
boot.matrix(BeckLee_mat50, bootstraps = 20, rarefaction = c(7, 10, 11))
## Bootstrapping an the matrix but without sampling Cimolestes and sampling Maelestes 10x more
boot.matrix(BeckLee_mat50, bootstraps = 20, prob = c("Cimolestes" = 0, "Maelestes" = 10))

## Bootstrapping a subsets of matrices
## Generating a dummy subsets of matrices
ordinated_matrix <- matrix(data = rnorm(90), nrow = 10, ncol = 9,
                           dimnames = list(letters[1:10]))
matrix_list <- custom.subsets(ordinated_matrix, list(A = 1:5, B = 6:10))
## Bootstrapping the subsets of matrices 20 times (each)
boot.matrix(matrix_list, bootstraps = 20)

}
\seealso{
\code{\link{cust.subsets}}, \code{\link{chrono.subsets}}, \code{\link{dispRity}}.
}
\author{
Thomas Guillerme
}
