% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dtrackr.R
\name{pivot_longer.trackr_df}
\alias{pivot_longer.trackr_df}
\title{Reshaping data using \code{tidyr::pivot_longer}}
\usage{
\method{pivot_longer}{trackr_df}(
  data,
  cols,
  ...,
  cols_vary = "fastest",
  names_to = "name",
  names_prefix = NULL,
  names_sep = NULL,
  names_pattern = NULL,
  names_ptypes = NULL,
  names_transform = NULL,
  names_repair = "check_unique",
  values_to = "value",
  values_drop_na = FALSE,
  values_ptypes = NULL,
  values_transform = NULL,
  .messages = c("long format", "{.count.in} before", "{.count.out} after"),
  .headline = .defaultHeadline()
)
}
\arguments{
\item{data}{A data frame to pivot.}

\item{cols}{<\code{\link[tidyr:tidyr_tidy_select]{tidy-select}}> Columns to pivot into
longer format.}

\item{...}{Additional arguments passed on to methods.}

\item{cols_vary}{When pivoting \code{cols} into longer format, how should the
output rows be arranged relative to their original row number?
\itemize{
\item \code{"fastest"}, the default, keeps individual rows from \code{cols} close
together in the output. This often produces intuitively ordered output
when you have at least one key column from \code{data} that is not involved in
the pivoting process.
\item \code{"slowest"} keeps individual columns from \code{cols} close together in the
output. This often produces intuitively ordered output when you utilize
all of the columns from \code{data} in the pivoting process.
}}

\item{names_to}{A character vector specifying the new column or columns to
create from the information stored in the column names of \code{data} specified
by \code{cols}.
\itemize{
\item If length 0, or if \code{NULL} is supplied, no columns will be created.
\item If length 1, a single column will be created which will contain the
column names specified by \code{cols}.
\item If length >1, multiple columns will be created. In this case, one of
\code{names_sep} or \code{names_pattern} must be supplied to specify how the
column names should be split. There are also two additional character
values you can take advantage of:
\itemize{
\item \code{NA} will discard the corresponding component of the column name.
\item \code{".value"} indicates that the corresponding component of the column
name defines the name of the output column containing the cell values,
overriding \code{values_to} entirely.
}
}}

\item{names_prefix}{A regular expression used to remove matching text
from the start of each variable name.}

\item{names_sep, names_pattern}{If \code{names_to} contains multiple values,
these arguments control how the column name is broken up.

\code{names_sep} takes the same specification as \code{\link[tidyr:separate]{separate()}}, and can either
be a numeric vector (specifying positions to break on), or a single string
(specifying a regular expression to split on).

\code{names_pattern} takes the same specification as \code{\link[tidyr:extract]{extract()}}, a regular
expression containing matching groups (\verb{()}).

If these arguments do not give you enough control, use
\code{pivot_longer_spec()} to create a spec object and process manually as
needed.}

\item{names_ptypes, values_ptypes}{Optionally, a list of column name-prototype
pairs. Alternatively, a single empty prototype can be supplied, which will
be applied to all columns. A prototype (or ptype for short) is a
zero-length vector (like \code{integer()} or \code{numeric()}) that defines the type,
class, and attributes of a vector. Use these arguments if you want to
confirm that the created columns are the types that you expect. Note that
if you want to change (instead of confirm) the types of specific columns,
you should use \code{names_transform} or \code{values_transform} instead.}

\item{names_transform, values_transform}{Optionally, a list of column
name-function pairs. Alternatively, a single function can be supplied,
which will be applied to all columns. Use these arguments if you need to
change the types of specific columns. For example, \code{names_transform = list(week = as.integer)} would convert a character variable called \code{week}
to an integer.

If not specified, the type of the columns generated from \code{names_to} will
be character, and the type of the variables generated from \code{values_to}
will be the common type of the input columns used to generate them.}

\item{names_repair}{What happens if the output has invalid column names?
The default, \code{"check_unique"} is to error if the columns are duplicated.
Use \code{"minimal"} to allow duplicates in the output, or \code{"unique"} to
de-duplicated by adding numeric suffixes. See \code{\link[vctrs:vec_as_names]{vctrs::vec_as_names()}}
for more options.}

\item{values_to}{A string specifying the name of the column to create
from the data stored in cell values. If \code{names_to} is a character
containing the special \code{.value} sentinel, this value will be ignored,
and the name of the value column will be derived from part of the
existing column names.}

\item{values_drop_na}{If \code{TRUE}, will drop rows that contain only \code{NA}s
in the \code{value_to} column. This effectively converts explicit missing values
to implicit missing values, and should generally be used only when missing
values in \code{data} were created by its structure.}

\item{.messages}{a set of glue specs. The glue code can use any global
variable, grouping variable, or \{.strata\}. Defaults to nothing.}

\item{.headline}{a headline glue spec. The glue code can use any global
variable, grouping variable, or \{.strata\}. Defaults to nothing.}
}
\value{
the result of the \code{tidyr::pivot_longer} but with a history graph
updated.
}
\description{
A drop in replacement for \code{\link[tidyr:pivot_longer]{tidyr::pivot_longer()}} which optionally takes a
message and headline to store in the history graph.
}
\examples{
library(dplyr)
library(dtrackr)

starwars \%>\%
  track() \%>\%
  tidyr::unnest(starships, keep_empty = TRUE) \%>\%
  tidyr::nest(world_data = c(-homeworld)) \%>\%
  history()

# There is a problem with `tidyr::unnest` that means if you want to override the
# `.messages` option at the moment it will most likely fail. Forcing the use of
# the specific `dtrackr::p_unnest` version solves this problem, until hopefully it is
# resolved in `tidyr`:
starwars \%>\%
  track() \%>\%
  p_unnest(
    films,
    .messages = c("{.count.in} characters", "{.count.out} appearances")
  ) \%>\%
  dplyr::group_by(gender) \%>\%
  tidyr::nest(
    people = c(-gender, -species, -homeworld),
    .messages = c("{.count.in} appearances", "{.count.out} planets")
  ) \%>\%
  status() \%>\%
  history()

# This example includes pivoting and nesting. The CMS patient care data
# has multiple tests per institution in a long format, and observed /
# denominator types. Firstly we pivot the data to allow us to easily calculate
# a total percentage for each institution. This is duplicated for every test
# so we nest the tests to get to one row per institution. Those institutions
# with invalid scores are excluded.
cms_history = tidyr::cms_patient_care \%>\%
  track() \%>\%
  tidyr::pivot_wider(names_from = type, values_from = score) \%>\%
  dplyr::mutate(
    percentage = sum(observed) / sum(denominator) * 100,
    .by = c(ccn, facility_name)
  ) \%>\%
  tidyr::nest(
    results = c(measure_abbr, observed, denominator),
    .messages = c("{.count.in} test results", "{.count.out} facilities")
  ) \%>\%
  exclude_all(
    percentage > 100 ~ "{.excluded} facilities with anomalous percentages",
    na.rm = TRUE
  )

print(cms_history \%>\% dtrackr::history())

# not run in examples:
if (interactive()) {
  cms_history \%>\% flowchart()
}
}
\seealso{
\code{\link[tidyr:pivot_longer]{tidyr::pivot_longer()}}
}
