% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/logger.R
\name{flog.logger}
\alias{flog.logger}
\alias{flog.trace}
\alias{flog.debug}
\alias{flog.info}
\alias{flog.warn}
\alias{flog.error}
\alias{flog.fatal}
\title{Manage loggers}
\arguments{
\item{msg}{The message to log}

\item{name}{The logger name to use}

\item{capture}{Capture print output of variables instead of interpolate}

\item{logger}{The logger to use. If \code{NULL} (the default), it is 
looked up based on \code{name}. Provide \code{logger} explicitely if 
the speed of the evaluation of log level is of concern (e.g., a 
\code{flog.trace} call in your function which has to be run many times).}

\item{\dots}{Optional arguments to populate the format string}

\item{expr}{An expression to evaluate}

\item{finally}{An optional expression to evaluate at the end}
}
\description{
Provides functions for writing log messages and managing loggers. Typically
only the flog.[trace|debug|info|warn|error|fatal] functions need to be used
in conjunction with flog.threshold to interactively change the log level.
}
\section{Usage}{

# Conditionally print a log statement at TRACE log level
flog.trace(msg, ..., name=flog.namespace(), logger=NULL, capture=FALSE)

# Conditionally print a log statement at DEBUG log level
flog.debug(msg, ..., name=flog.namespace(), logger=NULL, capture=FALSE)

# Conditionally print a log statement at INFO log level
flog.info(msg, ..., name=flog.namespace(), logger=NULL, capture=FALSE)

# Conditionally print a log statement at WARN log level
flog.warn(msg, ..., name=flog.namespace(), logger=NULL, capture=FALSE)

# Conditionally print a log statement at ERROR log level
flog.error(msg, ..., name=flog.namespace(), logger=NULL, capture=FALSE)

# Print a log statement at FATAL log level
flog.fatal(msg, ..., name=flog.namespace(), logger=NULL, capture=FALSE)

# Execute an expression and capture any warnings or errors
ftry(expr, error=stop, silent=FALSE, finally=NULL, details='')
}

\section{Additional Usage}{

These functions generally do not need to be called by an end user.

# Get the ROOT logger
flog.logger()

# Get the logger with the specified name
flog.logger(name)

# Set options for the given logger
flog.logger(name, threshold=NULL, appender=NULL, layout=NULL, carp=NULL)
}

\section{Details}{

These functions represent the high level interface to futile.logger.

The primary use case for futile.logger is to write out log messages. There
are log writers associated with all the predefined log levels: TRACE, DEBUG,
INFO, WARN, ERROR, FATAL. Log messages will only be written if the log level
is equal to or more urgent than the current threshold. By default the ROOT
logger is set to INFO.

> flog.debug("This won't print")
> flog.info("But this \%s", 'will')
> flog.warn("As will \%s", 'this')

Typically, the built in log level constants are used in the call, which
conform to the log4j levels (from least severe to most severe): TRACE,
DEBUG, INFO, WARN, ERROR, FATAL. It is not a strict requirement to use these
constants (any numeric value will work), though most users should find this
level of granularity sufficient.

Loggers are hierarchical in the sense that any requested logger that is
undefined will fall back to its most immediate defined parent logger. The
absolute parent is ROOT, which is guaranteed to be defined for the system
and cannot be deleted. This means that you can specify a new logger
directly.

> flog.info("This will fall back to 'my', then 'ROOT'", name='my.logger')

You can also change the threshold or any other setting associated with a
logger. This will create an explicit logger where any unspecified options
are copied from the parent logger.

> flog.appender(appender.file("foo.log"), name='my')
> flog.threshold(ERROR, name='my.logger')
> flog.info("This won't print", name='my.logger')
> flog.error("This %s print to a file", 'will', name='my.logger')

If you have a function which gets called many times, it is a good strategy 
to pass the logger directly instead of its name.

Instead of this:
> simulation_fun <- function(i) \{
>   flog.trace("We are in loop %d", i, name='my.logger')
>   i
> \}

... you can do this::
> my_logger <- flog.logger("my.logger")
> simulation_fun2 <- function(i) \{
>   flog.trace("We are in loop %d", i, logger=my_logger)
>   i
> \}

> system.time(for (i in 1:1000) simulation_fun(i))
> system.time(for (i in 1:1000) simulation_fun2(i))

If you define a logger that you later want to remove, use flog.remove.

The option 'capture' allows you to print out more complicated data
structures without a lot of ceremony. This variant doesn't accept format
strings and instead appends the value to the next line of output. Consider 

> m <- matrix(rnorm(12), nrow=3) \cr
> flog.info("Matrix:",m, capture=TRUE)

which preserves the formatting, whereas using capture=FALSE will have 
a cluttered output due to recycling.
}

\examples{

flog.threshold(DEBUG)
flog.debug("This debug message will print")

flog.threshold(WARN)
flog.debug("This one won't")

m <- matrix(rnorm(12), nrow=3)
flog.info("Matrix:",m, capture=TRUE)

ftry(log(-1))

\dontrun{
s <- c('FCX','AAPL','JPM','AMZN')
p <- TawnyPortfolio(s)

flog.threshold(TRACE,'tawny')
ws <- optimizePortfolio(p, RandomMatrixDenoiser())
z <- getIndexComposition()

flog.threshold(WARN,'tawny')
ws <- optimizePortfolio(p, RandomMatrixDenoiser())
z <- getIndexComposition()
}

\dontrun{
flog.appender(appender.modulo(1000), name='counter')
lapply(1:10000, function(i) flog.info("value is \%s",i, name='counter'))
}
}
\seealso{
\code{\link{flog.threshold}} \code{\link{flog.remove}}
\code{\link{flog.carp}} \code{\link{flog.appender}} \code{\link{flog.layout}}
}
\author{
Brian Lee Yung Rowe
}
\keyword{data}
