#include "RcppArmadillo.h"
#include <ggdmcHeaders/prior_type_casting.h>

//' Print a Joint Prior Distribution
//'
//' Displays the structure of a prior distribution specification passed to
//' C++ functions. This function is primarily intended for debugging and
//' inspection of the internal representation.
//'
//' @name print_prior
//'
//' @param p_prior_r A list specifying the prior distributions, typically
//' generated by \code{\link{BuildPrior}}.
//'
//' @return
//' A character vector summarising the prior specification.
//'
//' @details
//' This function is mainly used for debugging purposes. It provides a
//' readable summary of the prior distribution list as received by C++ code.
//' The input \code{p_prior_r} should be the output of \code{\link{BuildPrior}}.
//'
//' @examples
//' p0 <- c(A = 0.15, B = 0.45, mean_v = 2.25, sd_v = 0.15, t0 = 0.2)
//' p1 <- rep(0.1, 5)
//' p_prior <- BuildPrior(
//'     p0 = p0,
//'     p1 = p1,
//'     lower = rep(NA, 5),
//'     upper = rep(NA, 5),
//'     dist = rep("tnorm", 5),
//'     log_p = rep(TRUE, 5)
//' )
//' print_prior(p_prior)
//'
//' @export
// [[Rcpp::export]]
std::vector<std::string> print_prior(const Rcpp::List &p_prior_r)
{
    auto p_prior = new_prior(p_prior_r);
    std::vector<std::string> parameter_names = p_prior_r.names();

    p_prior->print(parameter_names);
    return parameter_names;
}

//' Density and Random Number Generation for a Join Prior Distribution
//'
//' \code{dprior} computes the log-density of a joint prior distribution at a 
//' given set of parameter values. \code{rprior} generates random samples from 
//' the same joint prior specification.
//'
//' @param p_prior_r A list specifying the prior distribution, typically 
//' constructed using \code{\link{BuildPrior}}.
//' @param parameters_r For \code{dprior}: A numeric vector of parameter values 
//' at which the prior log-density should be evaluated.
//' @param n For \code{rprior}: Integer specifying the number of random samples 
//' to generate (default is 1).
//'
//' @return
//' \describe{
//'   \item{\code{dprior}}{A numeric vector of log-density values.}
//'   \item{\code{rprior}}{A numeric matrix of dimension \code{n} × \code{nparameter}, 
//'     containing samples from the prior distribution. Each row is one sample.}
//' }
//'
//' @details
//' These functions implement the core computations for evaluating and sampling 
//' from a joint prior distribution specified via \code{\link{BuildPrior}}:
//' \itemize{
//'   \item \code{dprior}: Evaluates the log-density of the joint prior at the 
//'     given parameter values.
//'   \item \code{rprior}: Draws independent samples from the specified joint prior.
//' }
//'
//' The joint prior may include truncated normal, beta, gamma, and other common 
//' distributions, possibly bounded by user-specified lower and upper limits.
//'
//' @examples
//' p0 <- c(A = 0.15, B = 0.45, mean_v = 2.25, sd_v = 0.15, t0 = 0.2)
//' p1 <- rep(0.1, 5)
//' p_prior <- BuildPrior(
//'     p0 = p0,
//'     p1 = p1,
//'     lower = rep(NA, 5),
//'     upper = rep(NA, 5),
//'     dist = rep("tnorm", 5),
//'     log_p = rep(TRUE, 5)
//' )
//'
//' # Evaluate log-density
//' parameters <- seq(0.1, 0.5, by = 0.1)
//' res0 <- dprior(p_prior, parameters)
//' res1 <- dnorm(parameters, p0, 0.1, TRUE)
//' print(res0)
//' print(res1)
//'
//' # Generate samples from the prior
//' res2 <- rprior(p_prior, 1)
//' res3 <- rprior(p_prior, 2)
//' print(res2)
//' print(res3)
//'
//' @export
// [[Rcpp::export]]
std::vector<double> dprior(const Rcpp::List &p_prior_r,
                           const Rcpp::NumericVector &parameters_r)
{
    auto p_prior = new_prior(p_prior_r);
    auto parameters = Rcpp::as<std::vector<double>>(parameters_r);
    std::vector<double> out(parameters.size());
    p_prior->dprior(parameters, out);
    return out;
}

//' @rdname dprior
//' @export
// [[Rcpp::export]]
Rcpp::NumericMatrix rprior(const Rcpp::List &p_prior_r, unsigned int n = 1)

{
    auto p_ptr = new_prior(p_prior_r);
    arma::mat param_matrix(p_ptr->m_nparameter, n);

    for (size_t i = 0; i < n; ++i)
    {
        param_matrix.col(i) = p_ptr->rprior();
    }

    return Rcpp::wrap(param_matrix);
}

//' Sum of Log Prior Densities for a Joint Prior Distribution
//'
//' Computes the sum of log-densities for a vector of parameters, based on
//' their respective prior distribution specifications. This function is used
//' in Bayesian computations where the joint prior is the product of 
//' independent priors—thus, the log of the joint prior is the sum of
//' log-densities.
//'
//' @param p_prior_r A list of prior specifications. Each element is itself a 
//' list specifying the prior for one parameter, typically created by 
//' \code{\link{BuildPrior}}. Each sublist should contain:
//'   \itemize{
//'     \item \code{p0}: First parameter of the distribution (e.g., mean).
//'     \item \code{p1}: Second parameter (e.g., standard deviation or shape).
//'     \item \code{lower}: Lower bound for the parameter (if applicable).
//'     \item \code{upper}: Upper bound (if applicable).
//'     \item \code{dist}: Character string specifying the distribution type.
//'     \item \code{log_p}: Logical; should be \code{TRUE} for log-density computation.
//'   }
//'
//' @param parameters_r A numeric vector of parameter values at which to
//' evaluate the prior. Must be the same length as \code{p_prior_r}.
//'
//' @return
//' A single numeric value: the sum of log-densities evaluated at the given 
//' parameter vector.
//'
//' @details
//' This function performs the following steps:
//' \itemize{
//'   \item Iterates over each parameter and its associated prior specification
//'   \item Evaluates the log-density for each parameter
//'   \item Sums all log-densities to compute the joint log prior
//' }
//'
//' This is useful, for example, in computing the log-posterior of hierarchical 
//' Bayesian models where priors are assumed to be independent.
//'
//' @examples
//' p0 <- c(A = 0.15, B = 0.45, mean_v = 2.25, sd_v = 0.15, t0 = 0.2)
//' tnorm_prior <- BuildPrior(
//'     p0 = p0,
//'     p1 = rep(1, 5),
//'     lower = rep(0, 5),
//'     upper = rep(NA, 5),
//'     dist = rep("tnorm", 5),
//'     log_p = rep(TRUE, 5)
//' )
//'
//' npar <- length(tnorm_prior)
//' theta <- runif(npar, 0, 10)
//' result <- sumlogprior(p_prior_r = tnorm_prior, parameters_r = theta)
//' print(result)
//'
//' @export
// [[Rcpp::export]]
double sumlogprior(const Rcpp::List &p_prior_r,
                   const Rcpp::NumericVector &parameters_r)
{
    unsigned int nparameter0 = p_prior_r.size();
    unsigned int nparameter1 = parameters_r.size();
    if (nparameter0 != nparameter1)
    {
        Rcpp::stop("Parameter length mismatch:\n"
                   "  - p_prior_r contains " +
                   std::to_string(nparameter0) +
                   " elements\n"
                   "  - parameters_r contains " +
                   std::to_string(nparameter1) +
                   " elements\n"
                   "These vectors must have the same length.");
    }

    auto p_prior = new_prior(p_prior_r);
    arma::vec parameters = Rcpp::as<arma::vec>(parameters_r);
    return p_prior->sumlogprior(parameters);
}

/* -- The truncated normal distributions (external)-- */

//' Truncated Normal Distribution Functions
//'
//' Density, distribution function, and random number generation for the 
//' truncated normal distribution with mean \code{p0}, standard deviation 
//' \code{p1}, and truncation bounds \code{[lower, upper]}.
//'
//' @name tnorm
//' @aliases rtnorm ptnorm dtnorm
//'
//' @param n Integer. Number of random variates to generate (for \code{rtnorm}).
//' @param x Numeric vector of quantiles (for \code{dtnorm} and \code{ptnorm}).
//' @param p0 Mean of the underlying (untruncated) normal distribution.
//' @param p1 Standard deviation of the underlying normal distribution. Must be positive.
//' @param lower Lower bound of truncation (can be \code{-Inf}).
//' @param upper Upper bound of truncation (can be \code{Inf}).
//' @param lower_tail Logical; if \code{TRUE} (default), probabilities are
//' computed as \eqn{P[X \leq x]}, otherwise \eqn{P[X > x]}.
//' @param log_p Logical; if \code{TRUE}, probabilities or densities are returned on the log scale.
//'
//' @return
//' \describe{
//'   \item{\code{rtnorm}}{A numeric vector of length \code{n}, containing random variates from the truncated normal distribution.}
//'   \item{\code{ptnorm}}{A numeric vector of cumulative probabilities evaluated at \code{x}.}
//'   \item{\code{dtnorm}}{A numeric vector of (log) density values evaluated at \code{x}.}
//' }
//'
//' @details
//' These functions implement the truncated normal distribution:
//' \itemize{
//'   \item \code{rtnorm}: Generates random samples using inverse transform sampling.
//'   \item \code{ptnorm}: Computes the cumulative distribution function (CDF).
//'   \item \code{dtnorm}: Computes the probability density function (PDF).
//' }
//'
//' The truncated normal distribution is defined as:
//' \deqn{X \sim \mathcal{N}(\mu, \sigma^2), \quad \text{truncated to } [a, b]}
//' where \eqn{\mu = \code{p0}}, \eqn{\sigma = \code{p1}}, \eqn{a = \code{lower}}, and \eqn{b = \code{upper}}.
//'
//' Truncation can be one-sided (e.g., \code{lower = 0}, \code{upper = Inf}) or two-sided.
//'
//' @references
//' \itemize{
//'   \item Olmsted, J. (2020). \emph{RcppTN: Rcpp-Based Truncated Normal Distribution}. \url{https://github.com/olmjo/RcppTN}
//'   \item Jackson, C. (2011). \pkg{msm}: Multi-state Modelling. \url{https://cran.r-project.org/package=msm}
//'   \item Robert, C. P. (1995). "Simulation of truncated normal variables." \emph{Statistics and Computing}, 5(2), 121–125. \doi{10.1007/BF00143942}
//' }
//'
//' @examples
//' # Generate random samples from truncated normal
//' n <- 1e5
//' p0 <- 0
//' p1 <- 1
//' lower <- 0
//' upper <- Inf
//' rtnorm_dat <- rtnorm(n, p0, p1, lower, upper)
//'
//' # Density at quantiles
//' x <- seq(-5, 5, length.out = 1e3)
//' dtnorm_dat <- dtnorm(x, p0, p1, lower = -2, upper = 2, log_p = FALSE)
//'
//' # Cumulative probabilities
//' q <- seq(-5, 5, length.out = 1e3)
//' ptnorm_dat <- ptnorm(q, p0, p1, lower = -2, upper = 3, 
//'                      lower_tail = TRUE, log_p = FALSE)
//'
//' # Plotting
//' cex_lab <- 1
//' cex_axis <- 0.5
//' line_width <- 1.5
//'
//' hist(rtnorm_dat, breaks = "fd", freq = FALSE, xlab = "", 
//'      cex.lab = cex_lab, cex.axis = cex_axis, main = "")
//' plot(x, dtnorm_dat, type = "l", lwd = line_width, xlab = "",
//'      ylab = "Density", cex.lab = cex_lab, cex.axis = cex_axis, main = "")
//'
//' @export
// [[Rcpp::export]]
std::vector<double> rtnorm(unsigned int n, double p0, double p1, double lower,
                           double upper)
{
    std::vector<double> out(n);
    tnorm::tnorm_class tnorm_instance(p0, p1, lower, upper);
    std::generate(out.begin(), out.end(),
                  [&tnorm_instance]() { return tnorm_instance.r(); });
    return out;
}

//' @rdname tnorm
//' @export
// [[Rcpp::export]]
std::vector<double> ptnorm(std::vector<double> x, double p0, double p1,
                           double lower, double upper, bool lower_tail,
                           bool log_p = false)
{
    std::vector<double> out(x.size());
    tnorm::tnorm_class tnorm_instance(p0, p1, lower, upper, lower_tail, log_p);
    tnorm_instance.p(x, out);
    return out;
}

//' @rdname tnorm
//' @export
// [[Rcpp::export]]
std::vector<double> dtnorm(std::vector<double> x, double p0, double p1,
                           double lower, double upper, bool log_p = false)
{
    std::vector<double> out(x.size());
    tnorm::tnorm_class tnorm_instance(p0, p1, lower, upper, log_p);
    tnorm_instance.d(x, out);
    return out;
}
