% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gipsmult_find_MAP.R
\name{find_MAP}
\alias{find_MAP}
\title{Find the Maximum A Posteriori Estimation}
\usage{
find_MAP(
  g,
  max_iter = NA,
  optimizer = NA,
  show_progress_bar = TRUE,
  save_all_perms = FALSE,
  return_probabilities = FALSE
)
}
\arguments{
\item{g}{Object of a \code{gipsmult} class.}

\item{max_iter}{The number of iterations for an algorithm to perform.
At least 2. For \code{optimizer = "BF"}, it is not used;
for \code{optimizer = "MH"}, it has to be finite;
for \code{optimizer = "HC"}, it can be infinite.}

\item{optimizer}{The optimizer for the search of the maximum posteriori:
\itemize{
\item \code{"BF"} (the default for unoptimized \code{g} with \verb{perm size <= 9}) - Brute Force;
\item \code{"MH"} (the default for unoptimized \code{g} with \verb{perm size > 10}) - Metropolis-Hastings;
\item \code{"HC"} - Hill Climbing;
\item \code{"continue"} (the default for optimized \code{g}) - The same as
the \code{g} was optimized by (see Examples).
}

See the \strong{Possible algorithms to use as optimizers}
section below for more details.}

\item{show_progress_bar}{A boolean.
Indicate whether or not to show the progress bar:
\itemize{
\item When \code{max_iter} is infinite, \code{show_progress_bar} has to be \code{FALSE};
\item When \code{return_probabilities = TRUE}, then
shows an additional progress bar for the time
when the probabilities are calculated.
}}

\item{save_all_perms}{A boolean. \code{TRUE} indicates saving
a list of all permutations visited during optimization.
This can be useful sometimes but needs a lot more RAM.}

\item{return_probabilities}{A boolean. \code{TRUE} can only be provided
only when \code{save_all_perms = TRUE}. For:
\itemize{
\item \code{optimizer = "MH"} - use Metropolis-Hastings results to
estimate posterior probabilities;
\item \code{optimizer = "BF"} - use brute force results to
calculate exact posterior probabilities.
}

These additional calculations are costly, so a second and third
progress bar is shown (when \code{show_progress_bar = TRUE}).

To examine probabilities after optimization,
call \code{\link[=get_probabilities_from_gipsmult]{get_probabilities_from_gipsmult()}}.}
}
\value{
Returns an optimized object of a \code{gipsmult} class.
}
\description{
Use one of the optimization algorithms to find the permutation that
maximizes a posteriori probability based on observed data.
Not all optimization algorithms will always find the MAP, but they try
to find a significant value.
}
\details{
\code{find_MAP()} can produce a warning when:
\itemize{
\item the optimizer "hill_climbing" gets to the end of
its \code{max_iter} without converging.
\item the optimizer will find the permutation with smaller \code{n0} than
\code{number_of_observations}
}
}
\section{Possible algorithms to use as optimizers}{


For every algorithm, there are some aliases available.
\itemize{
\item \code{"brute_force"}, \code{"BF"}, \code{"full"} - use
the \strong{Brute Force} algorithm that checks the whole permutation
space of a given size. This algorithm will find
the actual Maximum A Posteriori Estimation, but it is
very computationally expensive for bigger spaces.
We recommend Brute Force only for \code{p <= 9}.
\item \code{"Metropolis_Hastings"}, \code{"MH"} - use
the \strong{Metropolis-Hastings} algorithm;
\href{https://en.wikipedia.org/wiki/Metropolis\%E2\%80\%93Hastings_algorithm}{see Wikipedia}.
The algorithm will draw a random transposition in every iteration
and consider changing the current state (permutation).
When the \code{max_iter} is reached, the algorithm will return the best
permutation calculated as the MAP Estimator.
This algorithm used in this context is a special case of the
\strong{Simulated Annealing} the user may be more familiar with;
\href{https://en.wikipedia.org/wiki/Simulated_annealing}{see Wikipedia}.
\item \code{"hill_climbing"}, \code{"HC"} - use
the \strong{hill climbing} algorithm;
\href{https://en.wikipedia.org/wiki/Hill_climbing}{see Wikipedia}.
The algorithm will check all transpositions in every iteration and
go to the one with the biggest a posteriori value.
The optimization ends when all \emph{neighbors} will have a smaller
a posteriori value. If the \code{max_iter} is reached before the end,
then the warning is shown, and it is recommended to continue
the optimization on the output of the \code{find_MAP()} with
\code{optimizer = "continue"}; see examples.
Remember that \code{p*(p-1)/2} transpositions will be checked
in every iteration. For bigger \code{p}, this may be costly.
}
}

\examples{
require("MASS") # for mvrnorm()

perm_size <- 6
mu1 <- runif(6, -10, 10)
mu2 <- runif(6, -10, 10) # Assume we don't know the means
sigma1 <- matrix(
  data = c(
    1.0, 0.8, 0.6, 0.4, 0.6, 0.8,
    0.8, 1.0, 0.8, 0.6, 0.4, 0.6,
    0.6, 0.8, 1.0, 0.8, 0.6, 0.4,
    0.4, 0.6, 0.8, 1.0, 0.8, 0.6,
    0.6, 0.4, 0.6, 0.8, 1.0, 0.8,
    0.8, 0.6, 0.4, 0.6, 0.8, 1.0
  ),
  nrow = perm_size, byrow = TRUE
)
sigma2 <- matrix(
  data = c(
    1.0, 0.5, 0.2, 0.0, 0.2, 0.5,
    0.5, 1.0, 0.5, 0.2, 0.0, 0.2,
    0.2, 0.5, 1.0, 0.5, 0.2, 0.0,
    0.0, 0.2, 0.5, 1.0, 0.5, 0.2,
    0.2, 0.0, 0.2, 0.5, 1.0, 0.5,
    0.5, 0.2, 0.0, 0.2, 0.5, 1.0
  ),
  nrow = perm_size, byrow = TRUE
)
# sigma1 and sigma2 are matrices invariant under permutation (1,2,3,4,5,6)
numbers_of_observations <- c(21, 37)
Z1 <- MASS::mvrnorm(numbers_of_observations[1], mu = mu1, Sigma = sigma1)
Z2 <- MASS::mvrnorm(numbers_of_observations[2], mu = mu2, Sigma = sigma2)
S1 <- cov(Z1)
S2 <- cov(Z2) # Assume we have to estimate the mean

g <- gipsmult(list(S1, S2), numbers_of_observations)

g_map <- find_MAP(g, max_iter = 5, show_progress_bar = FALSE, optimizer = "Metropolis_Hastings")
g_map

g_map2 <- find_MAP(g_map, max_iter = 5, show_progress_bar = FALSE, optimizer = "continue")

if (require("graphics")) {
  plot(g_map2, type = "both", logarithmic_x = TRUE)
}

g_map_BF <- find_MAP(g, show_progress_bar = FALSE, optimizer = "brute_force")
}
\seealso{
\itemize{
\item \code{\link[=gipsmult]{gipsmult()}} - The constructor of a \code{gipsmult} class.
The \code{gipsmult} object is used as the \code{g} parameter of \code{find_MAP()}.
\item \code{\link[=plot.gipsmult]{plot.gipsmult()}} - Practical plotting function for
visualizing the optimization process.
\item \code{\link[=get_probabilities_from_gipsmult]{get_probabilities_from_gipsmult()}} - When
\code{find_MAP(return_probabilities = TRUE)} was called,
probabilities can be extracted with this function.
\item \code{\link[=log_posteriori_of_gipsmult]{log_posteriori_of_gipsmult()}} - The function that the optimizers
of \code{find_MAP()} tries to find the argmax of.
}
}
