% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkw.R
\name{pgkw}
\alias{pgkw}
\title{Generalized Kumaraswamy Distribution CDF}
\usage{
pgkw(
  q,
  alpha = 1,
  beta = 1,
  gamma = 1,
  delta = 0,
  lambda = 1,
  lower.tail = TRUE,
  log.p = FALSE
)
}
\arguments{
\item{q}{Vector of quantiles (values generally between 0 and 1).}

\item{alpha}{Shape parameter \code{alpha} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{beta}{Shape parameter \code{beta} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{gamma}{Shape parameter \code{gamma} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{delta}{Shape parameter \code{delta} >= 0. Can be a scalar or a vector.
Default: 0.0.}

\item{lambda}{Shape parameter \code{lambda} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{lower.tail}{Logical; if \code{TRUE} (default), probabilities are
\eqn{P(X \le q)}, otherwise, \eqn{P(X > q)}.}

\item{log.p}{Logical; if \code{TRUE}, probabilities \eqn{p} are given as
\eqn{\log(p)}. Default: \code{FALSE}.}
}
\value{
A vector of probabilities, \eqn{F(q)}, or their logarithms if
\code{log.p = TRUE}. The length of the result is determined by the recycling
rule applied to the arguments (\code{q}, \code{alpha}, \code{beta},
\code{gamma}, \code{delta}, \code{lambda}). Returns \code{0} (or \code{-Inf}
if \code{log.p = TRUE}) for \code{q <= 0} and \code{1} (or \code{0} if
\code{log.p = TRUE}) for \code{q >= 1}. Returns \code{NaN} for invalid
parameters.
}
\description{
Computes the cumulative distribution function (CDF) for the five-parameter
Generalized Kumaraswamy (GKw) distribution, defined on the interval (0, 1).
Calculates \eqn{P(X \le q)}.
}
\details{
The cumulative distribution function (CDF) of the Generalized Kumaraswamy (GKw)
distribution with parameters \code{alpha} (\eqn{\alpha}), \code{beta}
(\eqn{\beta}), \code{gamma} (\eqn{\gamma}), \code{delta} (\eqn{\delta}), and
\code{lambda} (\eqn{\lambda}) is given by:
\deqn{
F(q; \alpha, \beta, \gamma, \delta, \lambda) =
  I_{x(q)}(\gamma, \delta+1)
}
where \eqn{x(q) = [1-(1-q^{\alpha})^{\beta}]^{\lambda}} and \eqn{I_x(a, b)}
is the regularized incomplete beta function, defined as:
\deqn{
I_x(a, b) = \frac{B_x(a, b)}{B(a, b)} = \frac{\int_0^x t^{a-1}(1-t)^{b-1} dt}{\int_0^1 t^{a-1}(1-t)^{b-1} dt}
}
This corresponds to the \code{\link[stats]{pbeta}} function in R, such that
\eqn{F(q; \alpha, \beta, \gamma, \delta, \lambda) = \code{pbeta}(x(q), \code{shape1} = \gamma, \code{shape2} = \delta+1)}.

The GKw distribution includes several special cases, such as the Kumaraswamy,
Beta, and Exponentiated Kumaraswamy distributions (see \code{\link{dgkw}} for details).
The function utilizes numerical algorithms for computing the regularized
incomplete beta function accurately, especially near the boundaries.
}
\examples{
\donttest{
# Simple CDF evaluation
prob <- pgkw(0.5, alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1) # Kw case
print(prob)

# Upper tail probability P(X > q)
prob_upper <- pgkw(0.5,
  alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1,
  lower.tail = FALSE
)
print(prob_upper)
# Check: prob + prob_upper should be 1
print(prob + prob_upper)

# Log probability
log <- pgkw(0.5,
  alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1,
  log.p = TRUE
)
print(log)
# Check: exp(log) should be prob
print(exp(log))

# Use of vectorized parameters
q_vals <- c(0.2, 0.5, 0.8)
alphas_vec <- c(0.5, 1.0, 2.0)
betas_vec <- c(1.0, 2.0, 3.0)
# Vectorizes over q, alpha, beta
pgkw(q_vals, alpha = alphas_vec, beta = betas_vec, gamma = 1, delta = 0.5, lambda = 0.5)

# Plotting the CDF for special cases
x_seq <- seq(0.01, 0.99, by = 0.01)
# Standard Kumaraswamy CDF
cdf_kw <- pgkw(x_seq, alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1)
# Beta distribution CDF equivalent (Beta(gamma, delta+1))
cdf_beta_equiv <- pgkw(x_seq, alpha = 1, beta = 1, gamma = 2, delta = 3, lambda = 1)
# Compare with stats::pbeta
cdf_beta_check <- stats::pbeta(x_seq, shape1 = 2, shape2 = 3 + 1)
# max(abs(cdf_beta_equiv - cdf_beta_check)) # Should be close to zero

plot(x_seq, cdf_kw,
  type = "l", ylim = c(0, 1),
  main = "GKw CDF Examples", ylab = "F(x)", xlab = "x", col = "blue"
)
lines(x_seq, cdf_beta_equiv, col = "red", lty = 2)
legend("bottomright",
  legend = c("Kw(2,3)", "Beta(2,4) equivalent"),
  col = c("blue", "red"), lty = c(1, 2), bty = "n"
)
}
}
\references{
Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation}

Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.
}
\seealso{
\code{\link{dgkw}}, \code{\link{qgkw}}, \code{\link{rgkw}},
\code{\link[stats]{pbeta}}
}
\author{
Lopes, J. E.
}
\keyword{cumulative}
\keyword{distribution}
