% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/price_indexes.R
\name{index_weights}
\alias{index_weights}
\title{Index weights}
\usage{
index_weights(
  type = c("Carli", "Jevons", "Coggeshall", "Dutot", "Laspeyres", "HybridLaspeyres",
    "LloydMoulton", "Palgrave", "Paasche", "HybridPaasche", "Drobisch", "Unnamed",
    "Tornqvist", "Walsh1", "Walsh2", "MarshallEdgeworth", "GearyKhamis", "Vartia1",
    "MontgomeryVartia", "Vartia2", "SatoVartia", "Theil", "Rao", "Lowe", "Young",
    "HybridCSWD")
)
}
\arguments{
\item{type}{The name of the index. See details for the possible types of
indexes.}
}
\value{
A function of current and base period prices/quantities that calculates
the relevant weights.
}
\description{
Calculate weights for a variety of different price indexes.
}
\details{
The \code{index_weights()} function returns a function to calculate weights
for a variety of price indexes. Weights for the following types of indexes
can be calculated.
\itemize{
\item Carli / Jevons / Coggeshall
\item Dutot
\item Laspeyres / Lloyd-Moulton
\item Hybrid Laspeyres (for use in a harmonic mean)
\item Paasche / Palgrave
\item Hybrid Paasche (for use in an arithmetic mean)
\item Törnqvist / Unnamed
\item Drobisch
\item Walsh-I (for an arithmetic Walsh index)
\item Walsh-II (for a geometric Walsh index)
\item Marshall-Edgeworth
\item Geary-Khamis
\item Montgomery-Vartia / Vartia-I
\item Sato-Vartia / Vartia-II
\item Theil
\item Rao
\item Lowe
\item Young
\item Hybrid-CSWD
}

The weights need not sum to 1, as this normalization isn't always
appropriate (i.e., for the Vartia-I weights).
}
\note{
Naming for the indexes and weights generally follows the CPI manual (2020),
Balk (2008), von der Lippe (2007), and Selvanathan and Rao (1994). In several
cases two or more names correspond to the same weights (e.g., Paasche and
Palgrave, or Sato-Vartia and Vartia-II). The calculations are given in the
examples.
}
\examples{
p1 <- price6[[2]]
p2 <- price6[[3]]
q1 <- quantity6[[2]]
q2 <- quantity6[[3]]
pb <- price6[[1]]
qb <- quantity6[[1]]

# Explicit calculation for each of the different weights
# Carli/Jevons/Coggeshall

all.equal(index_weights("Carli")(p2), rep(1, length(p1)))

# Dutot

all.equal(index_weights("Dutot")(p1), p1)

# Laspeyres / Lloyd-Moulton

all.equal(index_weights("Laspeyres")(p1, q1), p1 * q1)

# Hybrid Laspeyres

all.equal(index_weights("HybridLaspeyres")(p2, q1), p2 * q1)

# Paasche / Palgrave

all.equal(index_weights("Paasche")(p2, q2), p2 * q2)

# Hybrid Paasche

all.equal(index_weights("HybridPaasche")(p1, q2), p1 * q2)

# Tornqvist / Unnamed

all.equal(
  index_weights("Tornqvist")(p2, p1, q2, q1),
  0.5 * p1 * q1 / sum(p1 * q1) + 0.5 * p2 * q2 / sum(p2 * q2)
)

# Drobisch

all.equal(
  index_weights("Drobisch")(p2, p1, q2, q1),
  0.5 * p1 * q1 / sum(p1 * q1) + 0.5 * p1 * q2 / sum(p1 * q2)
)

# Walsh-I

all.equal(
  index_weights("Walsh1")(p1, q2, q1),
  p1 * sqrt(q1 * q2)
)

# Marshall-Edgeworth

all.equal(
  index_weights("MarshallEdgeworth")(p1, q2, q1),
  p1 * (q1 + q2)
)

# Geary-Khamis

all.equal(
  index_weights("GearyKhamis")(p1, q2, q1),
  p1 / (1 / q1 + 1 / q2)
)

# Montgomery-Vartia / Vartia-I

all.equal(
  index_weights("MontgomeryVartia")(p2, p1, q2, q1),
  logmean(p1 * q1, p2 * q2) / logmean(sum(p1 * q1), sum(p2 * q2))
)

# Sato-Vartia / Vartia-II

all.equal(
  index_weights("SatoVartia")(p2, p1, q2, q1),
  logmean(p1 * q1 / sum(p1 * q1), p2 * q2 / sum(p2 * q2))
)

# Walsh-II

all.equal(
  index_weights("Walsh2")(p2, p1, q2, q1),
  sqrt(p1 * q1 * p2 * q2)
)

# Theil

all.equal(index_weights("Theil")(p2, p1, q2, q1), {
  w0 <- scale_weights(p1 * q1)
  w1 <- scale_weights(p2 * q2)
  (w0 * w1 * (w0 + w1) / 2)^(1 / 3)
})

# Rao

all.equal(index_weights("Rao")(p2, p1, q2, q1), {
  w0 <- scale_weights(p1 * q1)
  w1 <- scale_weights(p2 * q2)
  w0 * w1 / (w0 + w1)
})

# Lowe

all.equal(index_weights("Lowe")(p1, qb), p1 * qb)

# Young

all.equal(index_weights("Young")(pb, qb), pb * qb)

# Hybrid CSWD (to approximate a CSWD index)

all.equal(index_weights("HybridCSWD")(p2, p1), sqrt(p1 / p2))

}
\references{
Balk, B. M. (2008). \emph{Price and Quantity Index Numbers}.
Cambridge University Press.

IMF, ILO, Eurostat, UNECE, OECD, and World Bank. (2020).
\emph{Consumer Price Index Manual: Concepts and Methods}.
International Monetary Fund.

von der Lippe, P. (2007). \emph{Index Theory and Price Statistics}. Peter Lang.

Selvanathan, E. A. and Rao, D. S. P. (1994).
\emph{Index Numbers: A Stochastic Approach}. MacMillan.
}
\seealso{
\code{\link[=update_weights]{update_weights()}} for price-updating weights.

\code{\link[=quantity_index]{quantity_index()}} to remap the arguments in these functions for a
quantity index.

Other price index functions: 
\code{\link{geks}()},
\code{\link{price_indexes}},
\code{\link{splice_index}()}
}
\concept{price index functions}
