% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MTDest.R
\name{MTDest}
\alias{MTDest}
\title{EM estimation of MTD parameters}
\usage{
MTDest(
  X,
  S,
  M = 0.01,
  init,
  iter = FALSE,
  nIter = 100,
  A = NULL,
  oscillations = FALSE
)
}
\arguments{
\item{X}{A vector or single-column data frame containing an MTD chain sample
(\code{X[1]} is the most recent).}

\item{S}{A numeric vector of distinct positive integers, sorted increasingly.
Typically, \code{S} represents a set of relevant lags.}

\item{M}{A stopping point for the EM algorithm. If \code{M=NULL} the algorithm
will run for a total of \code{nIter} iterations (i.e., no convergence check).}

\item{init}{A list with initial parameters: \code{p0} (optional), \code{lambdas}
(required), \code{pj} (required). The entries in \code{lambdas} are weights
for the distribution \code{p0} and the distributions present in the list
\code{pj}. Therefore, the order in which the elements appear in the vector
\code{lambdas} is important for correct assignment. See  \emph{Details}.}

\item{iter}{Logical. If \code{TRUE} include iteration diagnostics in the output
(number of updates \code{iterations}; vector of absolute log-likelihood changes
\code{deltaLogLik}; and \code{lastComputedDelta}, the last delta log-likelihood
compared against \code{M}).}

\item{nIter}{An integer positive number with the maximum number of EM iterations.}

\item{A}{Optional integer vector giving the state space. If omitted, defaults
to \code{sort(unique(X))}.}

\item{oscillations}{Logical. If \code{TRUE}, also compute oscillations for the
fitted model (see \code{\link{oscillation}}).}
}
\value{
An S3 object of class \code{c("MTDest", "MTD")} (a list) with at least the following elements:
\itemize{
\item \code{lambdas}: estimated mixture weights (independent part first, if any).
\item \code{pj}: list of estimated transition matrices \eqn{p_j}.
\item \code{p0}: estimated independent distribution (if applicable).
\item \code{logLik}: log-likelihood of the final fitted model.
\item \code{iterations}, \code{deltaLogLik}, \code{lastComputedDelta} (optional):
returned if \code{iter = TRUE}. Here, \code{iterations} is the number of
parameter updates performed; \code{deltaLogLik} stores the successive
absolute log-likelihood changes for the accepted updates; and
\code{lastComputedDelta} is the last computed change (which may be below
\code{M} when the loop stops by convergence).
\item \code{oscillations} (optional): returned if \code{oscillations = TRUE}.
\item \code{call}: the matched call.
\item \code{S}: the lag set used for estimation.
\item \code{A}: the state space used for estimation.
\item \code{n}: the sample size.
\item \code{n_eff}: the effective sample size used for estimation.
}
}
\description{
Estimation of MTD parameters through the Expectation Maximization (EM) algorithm.
}
\details{
Regarding the \code{M} parameter: it functions as a stopping
criterion within the EM algorithm. When the difference between
the log-likelihood computed with the newly estimated parameters
and that computed with the previous parameters falls below \code{M},
the algorithm halts. Nevertheless, if the value of \code{nIter}
(which represents the maximum number of iterations) is smaller
than the number of iterations required to meet the \code{M} criterion,
the algorithm will conclude its execution when \code{nIter} is reached.
To ensure that the \code{M} criterion is effectively utilized, we
recommend using a higher value for \code{nIter}, which is set to a
default of 100.

Concerning the \code{init} parameter, it is expected to be a list with 2 or
3 entries. These entries consist of:
an optional vector named \code{p0}, representing an independent
distribution (the probability in the first entry of \code{p0} must be
that of the smallest element in \code{A} and so on), a required list
of matrices \code{pj}, containing a stochastic matrix for each
element of \code{S} (the first matrix corresponds to the smallest lag in
\code{S} and so on), and a vector named \code{lambdas} representing
the weights, the first entry must be the weight for \code{p0}, and then one entry
for each element in \code{pj} list. If your MTD model does not have an independent
distribution \code{p0}, set \code{init$lambdas[1]=0}.
}
\section{Methods (S3)}{

Objects returned by \code{MTDest()} have class \code{c("MTDest","MTD")} and support:
\itemize{
\item \code{\link[base]{print}} and \code{\link[base]{summary}}: methods for fitted objects
(see \code{\link{MTDest-methods}}).
\item \code{\link[stats]{coef}}: extracts fitted parameters \code{lambdas}, \code{pj}, and \code{p0}
(inherited from \code{"MTD"}; see \code{\link{MTD-methods}}).
\item \code{\link[stats]{logLik}}: returns the final log-likelihood stored in the fit
(see \code{\link{MTDest-methods}}).
\item \code{\link[graphics]{plot}}: diagnostic plots for fitted objects (see \code{\link{plot.MTDest}}).
\item \code{\link{probs}}, \code{\link{oscillation}}, and \code{\link{perfectSample}}: additional utilities
available by inheritance from \code{"MTD"}.
\item \code{\link{as.MTD}}: coerces an \code{"MTDest"} fit to an \code{"MTD"} model.
}
}

\section{Accessors}{

Stable access to fitted components is provided by \code{\link{MTD-accessors}}, including
\code{\link{S}} (or \code{\link{Lambda}}), \code{\link{lags}}, \code{\link{lambdas}},
\code{\link{pj}}, \code{\link{p0}}, \code{\link{states}}, and \code{\link{transitP}}.
}

\examples{
# Simulating data.
set.seed(1)
MTD <- MTDmodel(Lambda = c(1, 10), A = c(0, 1), lam0 = 0.01)
X <- perfectSample(MTD, N = 400)

# Initial Parameters:
init <- list(
  'p0' = c(0.4, 0.6),
  'lambdas' = c(0.05, 0.45, 0.5),
  'pj' = list(
      matrix(c(0.2, 0.8, 0.45, 0.55), byrow = TRUE, ncol = 2),
      matrix(c(0.25, 0.75, 0.3, 0.7), byrow = TRUE, ncol = 2)
    )
 )

fit <- MTDest(X, S = c(1, 10), init = init, iter = TRUE)
str(fit, max.level = 1)
fit$logLik
class(fit)

fit2 <- MTDest(X, S = c(1, 10), init = init, oscillations = TRUE)
fit2$oscillations

}
\references{
Lebre, Sophie and Bourguignon, Pierre-Yves. (2008).
An EM algorithm for estimation in the Mixture Transition Distribution model.
\emph{Journal of Statistical Computation and Simulation}, \emph{78}(1), 1-15.
\doi{10.1080/00949650701266666}
}
\seealso{
\code{\link{MTDmodel}} for constructing an MTD model,
\code{\link{hdMTD}} for lag selection procedures,
\code{\link{MTDest-methods}} for methods specific to fitted objects,
\code{\link{MTD-methods}} for methods inherited from \code{"MTD"},
\code{\link{MTD-accessors}} for stable access to components,
and \code{\link{as.MTD}} for coercion of fits to model objects.
}
