#include <Rcpp.h>
using namespace Rcpp;

// Function that accepts parameter values for 3 main effects and two interaction
// and performs the sobel test. Main effects and covariance terms required.

//' Sobel test for the indirect effect - Two ordered mediator path
//'
//' This function takes the parameter estimates and covariances
//' and performs the Sobel test for two ordered mediators.
//'
//' @param mu1 Value of the estimate of the independent variable on the first mediator.
//' @param sig1 Value of the variance of the estimate of the independent variable on the first mediator.
//' @param mu2 Value of the estimate of the first mediator on the second mediator.  
//' @param sig2 Value of the variance of the estimate of the first mediator on the second mediator.  
//' @param mu3 Value of the estimate of the second mediator on the response.
//' @param sig3 Value of the variance of the estimate of the second mediator on the response.
//' @param mu2b Value of the estimate of the effect of the interaction of the independent and first mediator variable on the second mediator. 
//' @param sig2b Value of the variance of the estimate of the effect of the interaction of the independent and first mediator variable on the second mediator.
//' @param mu3b Value of the estimate of the effect of the interaction of the independent and second mediator variable on the response. 
//' @param sig3b Value of the variance of the estimate of the effect of the interaction of the independent and second mediator variable on the response.
//' @param sig12 Value of the covariance between mu1 and mu2.
//' @param sig12b Value of the covariance between mu1 and mu2b. 
//' @param sig13 Value of the covariance between mu1 and mu3. 
//' @param sig13b Value of the covariance between mu1 and mu3b.
//' @param sig22b Value of the covariance between mu2 and mu2b.
//' @param sig23 Value of the covariance between mu2 and mu3.
//' @param sig23b Value of the covariance between mu2 and mu2b.
//' @param sig2b3 Value of the covariance between mu2b and mu3. 
//' @param sig2b3b Value of the covariance between mu2b and mu3b.
//' @param sig33b Value of the covariance between mu3 and mu3b.
//' @param indL Value indicating the value of the independent variable used for the interaction. Typically 1.
//' @param mu1_0 Null value for mu1.
//' @param mu2_0 Null value for mu2.
//' @param mu3_0 Null value for mu3.
//' @param mu2b_0 Null value for mu2b.
//' @param mu3b_0 Null value for mu3b.
//' @returns A p-value for the test for the indirect effect.
//' @export
//' @examples
//' sobelTest_ord(1, .1, .25, .01, 0, 0, .15, .01, 0, 0)
// [[Rcpp::export]]
double sobelTest_ord(const long double & mu1, const long double & sig1,
                         const long double & mu2, const long double & sig2,
                         const long double & mu3, const long double & sig3,
                         const long double & mu2b = 0, const long double & sig2b = 0,
                         const long double & mu3b = 0, const long double & sig3b = 0,
                         const long double & sig12 = 0, const long double & sig12b = 0,
                         const long double & sig13 = 0, const long double & sig13b = 0,
                         const long double & sig22b = 0, const long double & sig23 = 0,
                         const long double & sig23b = 0,
                         const long double & sig2b3 = 0, const long double & sig2b3b = 0,
                         const long double & sig33b = 0,
                         const int & indL = 1,
                        double mu1_0 = 0, double mu2_0 = 0, double mu3_0 = 0, 
                        double mu2b_0 = 0, double mu3b_0 = 0){


  double g1 = (mu2 + mu2b*indL)*(mu3 + mu3b*indL);
  double g2 = mu1*(mu3 + mu3b*indL);
  double g2b = indL*g2;
  double g3 = mu1*(mu2 + mu2b*indL);
  double g3b = indL*g3;

  double p1  = g1*sig1   + g2*sig12  + g2b*sig12b  + g3*sig13  + g3b*sig13b;
  double p2  = g1*sig12  + g2*sig2   + g2b*sig22b  + g3*sig23  + g3b*sig23b;
  double p2b = g1*sig12b + g2*sig22b + g2b*sig2b   + g3*sig2b3 + g3b*sig2b3b;
  double p3  = g1*sig13  + g2*sig23  + g2b*sig2b3  + g3*sig3   + g3b*sig33b;
  double p3b = g1*sig13b + g2*sig23b + g2b*sig2b3b + g3*sig33b + g3b*sig3b;

  double deltaSE = std::sqrt( g1*p1 + g2*p2 + g2b*p2b + g3*p3 + g3b*p3b );

  double deltaEst = mu1*(mu2+indL*mu2b)*(mu3+indL*mu3b);
  double deltaNull = mu1_0*(mu2_0+indL*mu2b_0)*(mu3_0+indL*mu3b_0);
  
  return(R::pnorm(
    -fabs(
        (deltaEst - deltaNull) / deltaSE) , 0, 1, 1, 0)*2.0);
}
