% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bru.inference.R
\name{bru}
\alias{bru}
\alias{bru_rerun}
\alias{print.bru}
\title{Convenient model fitting using (iterated) INLA}
\usage{
bru(components = ~Intercept(1), ..., options = list(), .envir = parent.frame())

bru_rerun(result, options = list())

\method{print}{bru}(x, ...)
}
\arguments{
\item{components}{A \code{formula}-like specification of latent components.
Also used to define a default linear additive predictor.  See
\code{\link[=bru_comp]{bru_comp()}} for details.}

\item{...}{Obervation models, each constructed by a calling \code{\link[=bru_obs]{bru_obs()}}, or
named parameters that can be passed to a single \code{\link[=bru_obs]{bru_obs()}} call. Note that
all the arguments will be evaluated before calling \code{\link[=bru_obs]{bru_obs()}} in order to
detect if they are \code{like} objects. This means that special arguments that
need to be evaluated in the context of \code{response_data} or \code{data} (such as
\code{Ntrials}) may will only work that way in direct calls to \code{\link[=bru_obs]{bru_obs()}}.}

\item{options}{A \link{bru_options} options object or a list of options passed
on to \code{\link[=bru_options]{bru_options()}}}

\item{.envir}{Environment for component evaluation (for when a non-formula
specification is used)}

\item{result}{A previous estimation object of class \code{bru}}

\item{x}{A \code{bru} object to be printed}
}
\value{
bru returns an object of class "bru". A \code{bru} object inherits
from \code{INLA::inla} (see the inla documentation for its properties) and
adds additional information stored in the \code{bru_info} field.
}
\description{
This method is a wrapper for \code{INLA::inla} and provides
multiple enhancements.

\itemize{
\item
Easy usage of spatial covariates and automatic construction of inla
projection matrices for (spatial) SPDE models. This feature is
accessible via the \code{components} parameter. Practical examples on how to
use spatial data by means of the components parameter can also be found
by looking at the \link{lgcp} function's documentation.
\item
Constructing multiple likelihoods is straight forward. See \link{like} for
more information on how to provide additional likelihoods to \code{bru}
using the \code{...} parameter list.
\item
Support for non-linear predictors. See example below.
\item
Log Gaussian Cox process (LGCP) inference is
available by using the \code{cp} family or (even easier) by using the
\link{lgcp} function.
}
}
\section{Methods (by generic)}{
\itemize{
\item \code{print(bru)}: Print a summary of a \code{bru} object.

}}
\section{Functions}{
\itemize{
\item \code{bru_rerun()}: Continue the optimisation from a previously computed estimate. The estimation
\code{options} list can be given new values to override the original settings.

To rerun with a subset of the data (e.g. for cross validation or prior
sampling), use \code{\link[=bru_set_missing]{bru_set_missing()}} to set all or part of the response data
to \code{NA} before calling \code{bru_rerun()}.

}}
\examples{
\donttest{
if (bru_safe_inla()) {
  # Simulate some covariates x and observations y
  input.df <- data.frame(x = cos(1:10))
  input.df <- within(input.df, {
    y <- 5 + 2 * x + rnorm(10, mean = 0, sd = 0.1)
  })

  # Fit a Gaussian likelihood model
  fit <- bru(y ~ x + Intercept(1), family = "gaussian", data = input.df)

  # Obtain summary
  fit$summary.fixed
}


if (bru_safe_inla()) {
  # Alternatively, we can use the bru_obs() function to construct the likelihood:

  lik <- bru_obs(family = "gaussian",
              formula = y ~ x + Intercept,
              data = input.df)
  fit <- bru(~ x + Intercept(1), lik)
  fit$summary.fixed
}

# An important addition to the INLA methodology is bru's ability to use
# non-linear predictors. Such a predictor can be formulated via bru_obs()'s
# \code{formula} parameter. The z(1) notation is needed to ensure that
# the z component should be interpreted as single latent variable and not
# a covariate:

if (bru_safe_inla()) {
  z <- 2
  input.df <- within(input.df, {
    y <- 5 + exp(z) * x + rnorm(10, mean = 0, sd = 0.1)
  })
  lik <- bru_obs(
    family = "gaussian", data = input.df,
    formula = y ~ exp(z) * x + Intercept
  )
  fit <- bru(~ z(1) + Intercept(1), lik)

  # Check the result (z posterior should be around 2)
  fit$summary.fixed
}
}
}
\author{
Fabian E. Bachl \email{bachlfab@gmail.com}
}
