#' Generic plot function for bmeta object in jarbes.
#'
#'
#' This function is built with `ggplot2`. Parameters such as `binwidth`, `adjust`, `density.colors`, `density.lwd`, `density.lty`, and `density.alpha` correspond directly to arguments in the `ggplot2` geoms.
#'
#' @param x The object generated by the bmeta function.
#' @param n.samples The number of posterior samples to plot.
#' @param x.lim Numeric vector of length 2 specifying the x-axis limits.
#' @param y.lim Numeric vector of length 2 specifying the y-axis limits.
#' @param x.lab Text with the label of the x-axis.
#' @param y.lab Text with the label of the y-axis.
#' @param title.plot Text for setting a title in the plot.
#' @param show.legend A logical value indicating whether to show the legend.
#' @param legend.labels A character vector of length 2 to customize the legend labels.
#' @param legend.title A character value to customize the legend title.
#' @param show.histogram A logical value indicating whether to show the histogram.
#' @param show.rug A logical value indicating whether to show the rug plot.
#' @param show.vlines A logical value indicating whether to show the vertical lines for the means.
#' @param binwidth A numeric value specifying the bin width for the histogram.
#' @param adjust A numeric value to adjust the bandwidth of the density curves. Defaults to 1.
#' @param density.colors A character vector of length 2 for the density curve colors.
#' @param density.lwd A numeric vector of length 2 for the density curve line widths.
#' @param density.lty A numeric vector of length 2 for the density curve line types.
#' @param density.alpha A numeric vector of length 2 for the density curve transparencies.
#' @param hist.color A character value for the histogram border color.
#' @param hist.fill A character value for the histogram fill color.
#' @param rug.color A character value for the rug plot color.
#' @param rug.alpha A numeric value between 0 and 1 for the rug plot transparency.
#'
#' @param ... \dots
#'
#' @import ggplot2
#'
#' @export
plot.bmeta = function(x,
                      n.samples = NULL,
                      x.lim = NULL,
                      y.lim = NULL,
                      x.lab = expression(mu[theta]),
                      y.lab = "Posterior Distribution",
                      title.plot = "Normal RE",
                      show.legend = TRUE,
                      legend.labels = c("Posterior of the difference in means", "Posterior Predictive Distribution"),
                      legend.title = "Distribution",
                      show.histogram = TRUE,
                      show.rug = TRUE,
                      show.vlines = TRUE,
                      binwidth = 0.08,
                      adjust = 1,
                      density.colors = c("blue", "red"),
                      density.lwd = c(1, 1),
                      density.lty = c("solid", "dashed"),
                      density.alpha = c(0.2, 0.2),
                      hist.color = "black",
                      hist.fill = "grey95",
                      rug.color = "green",
                      rug.alpha = 0.5,
                      ...) {

  mu = mu.new = type = after_stat = NULL

  # Use all samples if n.samples is not specified
  if (is.null(n.samples)) {
    n.samples = length(x$BUGSoutput$sims.list$mu)
  }

  # Create a data frame for the posterior distribution (for histogram and density)
  posterior_data = data.frame(mu = x$BUGSoutput$sims.list$mu[1:n.samples])

  # Create a data frame for the posterior predictive distribution (for density)
  predictive_data = data.frame(mu.new = x$BUGSoutput$sims.list$mu.new[1:n.samples])

  # Set default x.lim if not specified by the user
  if (is.null(x.lim)) {
    q1 = quantile(posterior_data$mu, 0.25, na.rm = TRUE)
    q3 = quantile(posterior_data$mu, 0.75, na.rm = TRUE)
    iqr = q3 - q1
    lower_lim = q1 - 8 * iqr
    upper_lim = q3 + 8 * iqr
    x.lim = c(lower_lim, upper_lim)
  }


  # Create the plot
  plot1 = ggplot()

  # Add histogram
  if (show.histogram) {
    plot1 = plot1 + geom_histogram(data = posterior_data, aes(x = mu, y = after_stat(density)),
                                   binwidth = binwidth,
                                   colour = hist.color,
                                   fill = hist.fill)
  }

  # Add density curves and map aesthetics
  plot1 = plot1 +
    geom_density(data = posterior_data, aes(x = mu, color = legend.labels[1], linetype = legend.labels[1]),
                 adjust = adjust,
                 linewidth = density.lwd[1],
                 alpha = density.alpha[1]) +
    geom_density(data = predictive_data, aes(x = mu.new, color = legend.labels[2], linetype = legend.labels[2]),
                 adjust = adjust,
                 linewidth = density.lwd[2],
                 alpha = density.alpha[2])

  # Add rug plot
  if (show.rug) {
    plot1 = plot1 + geom_rug(data = data.frame(TE = x$data$TE), aes(x = TE), sides = "b",
                             lwd = 1, col = rug.color, alpha = rug.alpha, show.legend = FALSE)
  }

  plot1 = plot1 +
    coord_cartesian(xlim = x.lim, ylim = y.lim) +
    xlab(x.lab) +
    ylab(y.lab) +
    geom_vline(aes(xintercept = 0), lty = 2, color = "black")

  if (show.vlines) {
    plot1 = plot1 +
      geom_vline(aes(xintercept = mean(x$BUGSoutput$sims.list$mu[1:n.samples])), linetype = "dashed", color = density.colors[1]) +
      geom_vline(aes(xintercept = mean(x$BUGSoutput$sims.list$mu.new[1:n.samples])), linetype = "dashed", color = density.colors[2])
  }

  plot1 = plot1 +
    theme_bw() +
    ggtitle(title.plot)

  # Manually set color and linetype scales for the legend
  plot1 = plot1 +
    scale_color_manual(name = legend.title,
                       values = setNames(density.colors, legend.labels)) +
    scale_linetype_manual(name = legend.title,
                          values = setNames(density.lty, legend.labels)) +
    theme(legend.position = if (show.legend) "right" else "none")

  return(suppressWarnings(plot1))
}
