% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.R
\name{geom_line_mcc}
\alias{geom_line_mcc}
\title{Add Reference Lines at an MCC Threshold to ggplot2 Objects}
\usage{
geom_line_mcc(
  mcc_object,
  threshold = 1,
  linetype = 2,
  color = NULL,
  alpha = 0.7,
  linewidth = 0.5,
  show_labels = FALSE,
  label_size = 3,
  label_nudge_x = 0,
  label_nudge_y = 0.05
)
}
\arguments{
\item{mcc_object}{An object of class \code{mcc} containing MCC estimates.}

\item{threshold}{numeric;determines MCC value threshold to use (default =
\code{1.0})}

\item{linetype}{Line type for the reference lines. Default is \code{2} (dashed).
Can be numeric (1-6) or character ("solid", "dashed", "dotted", etc.).}

\item{color}{Color for the reference lines. If \code{NULL} (default), uses gray.}

\item{alpha}{Transparency level for the reference lines. Default is \code{0.7}.}

\item{linewidth}{Width of the reference lines. Default is \code{0.5}.}

\item{show_labels}{Logical indicating whether to add text labels at the
intersection points. Default is \code{FALSE}.}

\item{label_size}{Size of the text labels if \code{show_labels = TRUE}. Default is \code{3}.}

\item{label_nudge_x}{Horizontal offset for labels. Default is \code{0}.}

\item{label_nudge_y}{Vertical offset for labels. Default is \code{0.05}.}
}
\value{
A \code{ggplot2} layer object that can be added to a ggplot using the \code{+}
operator.
}
\description{
Adds horizontal and vertical reference lines to mark when the Mean Cumulative
Count (MCC) reaches the \code{threshold}. This function returns a list of \code{ggplot2}
geoms that can be added to existing plots using the \code{+} operator. For
grouped analyses, it creates separate reference lines for each group.
}
\details{
This function identifies the time when MCC first reaches or exceeds the
specified MCC \code{threshold}. It then creates:
\itemize{
\item A horizontal line from x = 0 to the time when MCC = \code{threshold}
\item A vertical line from y = 0 to MCC = \code{threshold} at that time point
}

For grouped analyses, separate reference lines are created for each group
that reaches MCC = \code{threshold}. Groups that never reach MCC = \code{threshold}
will not have reference lines added.

The function is designed to work seamlessly with the existing \code{plot.mcc()}
method and can be chained using ggplot2's \code{+} syntax.
}
\examples{
# Create sample data
library(dplyr)
df <- data.frame(
  id = c(1, 2, 3, 4, 4, 4, 4, 5, 5),
  time = c(8, 1, 5, 2, 6, 7, 8, 3, 3),
  cause = c(0, 0, 2, 1, 1, 1, 0, 1, 2),
  group = c("A", "A", "B", "B", "B", "B", "B", "A", "A")
) |>
  arrange(id, time)

# Ungrouped analysis
mcc_overall <- mcc(df, "id", "time", "cause")

# Basic plot with reference lines
plot(mcc_overall) +
  geom_line_mcc(mcc_overall) +
  labs(title = "MCC with Reference Lines at 1.0")

# Grouped analysis
mcc_grouped <- mcc(df, "id", "time", "cause", by = "group")

# Plot with group-specific reference lines
plot(mcc_grouped) +
  geom_line_mcc(mcc_grouped, linetype = "dotted", alpha = 0.8) +
  labs(title = "Grouped MCC with Reference Lines")

# With labels
plot(mcc_overall) +
  geom_line_mcc(mcc_overall, show_labels = TRUE, color = "red") +
  labs(title = "MCC with Labeled Reference Lines")

# Clean up
rm(df, mcc_overall, mcc_grouped)
}
