% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mies_methods.R
\name{mies_generation_apply}
\alias{mies_generation_apply}
\title{Aggregate Values for All Generations Present}
\usage{
mies_generation_apply(
  archive,
  fitness_aggregator,
  include_previous_generations = FALSE
)
}
\arguments{
\item{archive}{(\code{\link[bbotk:Archive]{Archive}})\cr
The archive over which to aggregate.}

\item{fitness_aggregator}{(\code{function})\cr
Aggregation function, called with information about alive individuals of each generation. See \code{\link[=mies_aggregate_single_generation]{mies_aggregate_single_generation()}}.}

\item{include_previous_generations}{(\code{logical(1)})\cr
Aggregate all individuals that were alive at \code{generation} or at any point before that.
Duplicates with the same \code{x_id} are removed, meaning that if an individual was re-evaluated with different fidelity, only the last
re-evaluation is counted.
However, note that individuals from different generations may still have been evaluated with different fidelity, so if
Default \code{FALSE}.}
}
\value{
\code{data.table} with columns \code{dob}, next to the columns constructed from the return values of \code{fitness_aggregator}.
}
\description{
Applies a \code{fitness_aggregator} function to the values that were alive in the archive at at any generation.
\code{\link[=mies_aggregate_single_generation]{mies_aggregate_single_generation()}} is used, see there for more information about \code{fitness_aggregator}.

Generations for which \code{fitness_aggregator} returns \code{NULL}, or which are not present in any \code{dob} in the archive,
or which contain no alive individuals (e.g. because \code{eol} is smaller or equal \code{dob} for all of them) are ignored.

\code{as.list()} is applied to the values returned by \code{fitness_aggregator}, and \code{\link[data.table:rbindlist]{data.table::rbindlist()}} is called on
the list of resulting values. If the first non-\code{NULL}-value returned by \code{fitness_aggregator}, then \code{\link[data.table:rbindlist]{data.table::rbindlist()}}
is called with \code{fill = TRUE} and \code{use.names = TRUE}.

If no non-empty generations are present, or \code{fitness_aggregator} returns \code{NULL} on every call, then the return value
is \code{data.table(dob = numeric(0))}.

In contrast with \code{\link[=mies_aggregate_generations]{mies_aggregate_generations()}}, \code{mies_generate_apply()} can construct aggregated values for
entire fitness matrices, not only individual objectives (see examples). However, \code{\link[=mies_aggregate_generations]{mies_aggregate_generations()}} is simpler
if per-objective aggregates are desired.
}
\examples{
set.seed(1)
library("bbotk")
lgr::threshold("warn")

objective <- ObjectiveRFun$new(
  fun = function(xs) {
    list(y1 = xs$x1, y2 = xs$x2)
  },
  domain = ps(x1 = p_dbl(0, 1), x2 = p_dbl(-1, 0)),
  codomain = ps(y1 = p_dbl(0, 1, tags = "maximize"),
    y2 = p_dbl(-1, 0, tags = "minimize"))
)
oi <- OptimInstanceMultiCrit$new(objective,
  terminator = trm("evals", n_evals = 40))

op <- opt("mies",
  lambda = 4, mu = 4,
  mutator = mut("gauss", sdev = 0.1),
  recombinator = rec("xounif"),
  parent_selector = sel("random"),
  survival_selector = sel("best", scl("hypervolume"))
)

op$optimize(oi)

# Aggregated hypervolume of individuals alive in each gen:
mies_generation_apply(oi$archive, function(fitnesses) {
  domhv(fitnesses)
})

# Aggregated hypervolume of all points evaluated up to each gen
# (may be slightly more, since the domhv of more points is evaluated).
# This would be the dominated hypervolume of the result set at each
# generation:
mies_generation_apply(oi$archive, function(fitnesses) {
  domhv(fitnesses)
}, include_previous_generations = TRUE)

# The following are simpler with mies_aggregate_single_generations():
mies_generation_apply(oi$archive, function(fitnesses) {
  apply(fitnesses, 2, mean)
})
# Compare:
mies_aggregate_generations(oi, aggregations = list(mean = mean))

mies_generation_apply(oi$archive, function(objectives_unscaled) {
  apply(objectives_unscaled, 2, mean)
})
# Compare:
mies_aggregate_generations(oi, aggregations = list(mean = mean),
  as_fitnesses = FALSE)
}
\concept{aggregation functions}
