% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/missoNet.R
\name{missoNet}
\alias{missoNet}
\title{Fit missoNet models with missing responses}
\usage{
missoNet(
  X,
  Y,
  rho = NULL,
  GoF = "eBIC",
  lambda.beta = NULL,
  lambda.theta = NULL,
  lambda.beta.min.ratio = NULL,
  lambda.theta.min.ratio = NULL,
  n.lambda.beta = NULL,
  n.lambda.theta = NULL,
  beta.pen.factor = NULL,
  theta.pen.factor = NULL,
  penalize.diagonal = NULL,
  beta.max.iter = 10000,
  beta.tol = 1e-05,
  theta.max.iter = 10000,
  theta.tol = 1e-05,
  eta = 0.8,
  eps = 1e-08,
  standardize = TRUE,
  standardize.response = TRUE,
  relax.net = FALSE,
  adaptive.search = FALSE,
  parallel = FALSE,
  cl = NULL,
  verbose = 1
)
}
\arguments{
\item{X}{Numeric matrix (\eqn{n \times p}). Predictors (no missing values).}

\item{Y}{Numeric matrix (\eqn{n \times q}). Responses, may contain \code{NA}/\code{NaN}.}

\item{rho}{Optional numeric vector of length \eqn{q}. Working missingness
probabilities; if \code{NULL} (default), estimated from \code{Y}.}

\item{GoF}{Character. Goodness-of-fit criterion: \code{"AIC"}, \code{"BIC"},
or \code{"eBIC"} (default).}

\item{lambda.beta, lambda.theta}{Optional numeric vectors (or scalars).
Candidate regularization paths for \eqn{\mathbf{B}} and \eqn{\Theta}.
If \code{NULL}, paths are generated automatically.}

\item{lambda.beta.min.ratio, lambda.theta.min.ratio}{Optional numerics in \code{(0,1]}.
Ratio of the smallest to largest lambda when generating paths (ignored if the
corresponding \code{lambda.*} is supplied).}

\item{n.lambda.beta, n.lambda.theta}{Optional integers. Lengths of automatically
generated lambda paths (ignored if the corresponding \code{lambda.*} is supplied).}

\item{beta.pen.factor}{Optional \eqn{p \times q} non-negative matrix of element-wise
penalty multipliers for \eqn{\mathbf{B}}. \code{Inf} = maximum penalty;
\code{0} = no penalty for that coefficient. Default: all 1s (equal penalty).}

\item{theta.pen.factor}{Optional \eqn{q \times q} non-negative matrix of element-wise
penalty multipliers for \eqn{\Theta}. Off-diagonal entries control edge
penalties; diagonal treatment is governed by \code{penalize.diagonal}.
\code{Inf} = maximum penalty; \code{0} = no penalty for that coefficient.
Default: all 1s (equal penalty).}

\item{penalize.diagonal}{Logical or \code{NULL}. Whether to penalize the diagonal
of \eqn{\Theta}. If \code{NULL} (default) the choice is made automatically.}

\item{beta.max.iter, theta.max.iter}{Integers. Max iterations for the
\eqn{\mathbf{B}} update (FISTA) and \eqn{\Theta} update (graphical lasso).
Defaults: \code{10000}.}

\item{beta.tol, theta.tol}{Numerics \eqn{> 0}. Convergence tolerances for the
\eqn{\mathbf{B}} and \eqn{\Theta} updates. Defaults: \code{1e-5}.}

\item{eta}{Numeric in \code{(0,1)}. Backtracking line-search parameter for the
\eqn{\mathbf{B}} update (default \code{0.8}).}

\item{eps}{Numeric in \code{(0,1)}. Eigenvalue floor used to stabilize positive
definiteness operations (default \code{1e-8}).}

\item{standardize}{Logical. Standardize columns of \code{X} internally? Default \code{TRUE}.}

\item{standardize.response}{Logical. Standardize columns of \code{Y} internally?
Default \code{TRUE}.}

\item{relax.net}{(Experimental) Logical. If \code{TRUE}, refit active edges of \eqn{\Theta}
without \eqn{\ell_1} penalty (de-biased network). Default \code{FALSE}.}

\item{adaptive.search}{(Experimental) Logical. Use adaptive two-stage lambda search? Default \code{FALSE}.}

\item{parallel}{Logical. Evaluate parts of the grid in parallel using a provided
cluster? Default \code{FALSE}.}

\item{cl}{Optional cluster from \code{parallel::makeCluster()} (required if
\code{parallel = TRUE}).}

\item{verbose}{Integer in \code{0,1,2}. \code{0} = silent, \code{1} = progress (default),
\code{2} = detailed tracing (not supported in parallel mode).}
}
\value{
A list of class \code{"missoNet"} with components:
\describe{
\item{est.min}{List at the selected lambda pair:
\code{Beta} (\eqn{p \times q}), \code{Theta} (\eqn{q \times q}),
intercept \code{mu} (length \eqn{q}), \code{lambda.beta}, \code{lambda.theta},
\code{lambda.beta.idx}, \code{lambda.theta.idx}, scalar \code{gof}
(AIC/BIC/eBIC at optimum), and (if requested) \code{relax.net}.}
\item{rho}{Length-\eqn{q} vector of working missingness probabilities.}
\item{lambda.beta.seq, lambda.theta.seq}{Unique lambda values explored along
the grid for \eqn{\mathbf{B}} and \eqn{\Theta}.}
\item{penalize.diagonal}{Logical indicating whether the diagonal of
\eqn{\Theta} was penalized.}
\item{beta.pen.factor, theta.pen.factor}{Penalty factor matrices actually used.}
\item{param_set}{List with fitting diagnostics:
\code{n}, \code{p}, \code{q}, \code{standardize}, \code{standardize.response},
the vector of criterion values \code{gof}, and the evaluated grids
\code{gof.grid.beta}, \code{gof.grid.theta} (length equals number of fitted models).}
}
}
\description{
Fit a penalized multi-task regression with a response-network (\eqn{\Theta})
under missing responses. The method jointly estimates the coefficient matrix
\eqn{\mathbf{B}} and the precision matrix \eqn{\Theta} via penalized
likelihood with \eqn{\ell_1} penalties on \eqn{\mathbf{B}} and the off-diagonal
entries of \eqn{\Theta}.
}
\details{
The conditional Gaussian model is
\deqn{Y_i = \mu + X_i \mathbf{B} + E_i, \qquad E_i \sim \mathcal{N}_q(0, \Theta^{-1}).}

where:
\itemize{
\item \eqn{Y_i} is the \eqn{i}-th observation of \eqn{q} responses
\item \eqn{X_i} is the \eqn{i}-th observation of \eqn{p} predictors
\item \eqn{\mathbf{B}} is the \eqn{p \times q} coefficient matrix
\item \eqn{\Theta} is the \eqn{q \times q} precision matrix
\item \eqn{\mu} is the intercept vector
}

The parameters are estimated by solving:
\deqn{\min_{\mathbf{B}, \Theta \succ 0} \quad g(\mathbf{B}, \Theta) + 
  \lambda_B \|\mathbf{B}\|_1 + \lambda_\Theta \|\Theta\|_{1,\mathrm{off}}}

where \eqn{g} is the negative log-likelihood.

Missing values in \code{Y} are accommodated through unbiased estimating equations
using column-wise observation probabilities. Internally, \code{X} and \code{Y}
may be standardized for numerical stability; returned estimates are re-scaled
back to the original units.

The grid search spans \code{lambda.beta} and \code{lambda.theta}. The optimal
pair is selected by the user-chosen goodness-of-fit criterion \code{GoF}:
\code{"AIC"}, \code{"BIC"}, or \code{"eBIC"} (default). If
\code{adaptive.search = TRUE}, a two-stage pre-optimization narrows the grid
before the main search (faster on large problems, with a small risk of missing
the global optimum).
}
\examples{
sim <- generateData(n = 120, p = 10, q = 6, rho = 0.1)
X <- sim$X; Y <- sim$Z

\donttest{
# Fit with defaults (criterion = eBIC)
fit1 <- missoNet(X, Y)
# Extract the optimal estimates
Beta.hat <- fit1$est.min$Beta
Theta.hat <- fit1$est.min$Theta

# Plot missoNet results
plot(fit1, type = "heatmap")
plot(fit1, type = "scatter")

# Provide short lambda paths
fit2 <- missoNet(
  X, Y,
  lambda.beta  = 10^seq(0, -2, length.out = 5),
  lambda.theta = 10^seq(0, -2, length.out = 5),
  GoF = "BIC"
)

# Test single lambda choice
fit3 <- missoNet(
  X, Y,
  lambda.beta  = 0.1,
  lambda.theta = 0.1,
)

# De-biased network on the active set
fit4 <- missoNet(X, Y, relax.net = TRUE, verbose = 0)

# Adaptive search for large problems
fit5 <- missoNet(X = X, Y = Y, adaptive.search = TRUE, verbose = 0)

# Parallel (requires a cluster)
library(parallel)
cl <- makeCluster(2)
fit_par <- missoNet(X, Y, parallel = TRUE, cl = cl, verbose = 0)
stopCluster(cl)
}

}
\references{
Zeng, Y., et al. (2025). \emph{Multivariate regression with missing response data
for modelling regional DNA methylation QTLs}. arXiv:2507.05990.
}
\seealso{
\code{\link{cv.missoNet}} for cross-validated selection;
generic methods such as \code{plot()} and \code{predict()} for objects of class
\code{"missoNet"}.
}
\author{
Yixiao Zeng \email{yixiao.zeng@mail.mcgill.ca}, Celia M. T. Greenwood
}
