% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom_km.R
\name{geom_km}
\alias{geom_km}
\title{Kaplan-Meier curves of survival data}
\usage{
geom_km(
  network,
  ...,
  transform = c("identity", "cloglog", "log", "cumhaz"),
  curve_args = list(),
  cens_args = list()
)
}
\arguments{
\item{network}{A \code{nma_data} network object containing survival outcomes}

\item{...}{Additional arguments passed to \code{\link[survival:survfit]{survival::survfit()}}}

\item{transform}{Character string giving the transformation to apply to the
KM curves before plotting. The default is \code{"identity"} for no
transformation; other options are \code{"cloglog"} for \eqn{\log(-\log(S))},
\code{"log"} for \eqn{\log(S)}, or \code{"cumhaz"} for the cumulative hazard
\eqn{-\log(S)}.}

\item{curve_args}{Optional list of arguments to customise the curves plotted
with \code{\link[ggplot2:geom_path]{ggplot2::geom_step()}}}

\item{cens_args}{Optional list of arguments to customise the censoring marks
plotted with \code{\link[ggplot2:geom_point]{ggplot2::geom_point()}}}
}
\value{
A ggplot2 geom list that can be added to a ggplot2 plot object
}
\description{
This helper function constructs a ggplot2 geom to plot Kaplan-Meier curves
from a network containing survival or time-to-event outcomes. This is useful
for overlaying the "raw" survival data on the estimated survival functions
created with plotted with \code{\link[=plot.surv_nma_summary]{plot.surv_nma_summary()}}, but can also be used
standalone to plot Kaplan-Meier curves before fitting a model.
}
\examples{
# Set up newly-diagnosed multiple myeloma network

head(ndmm_ipd)
head(ndmm_agd)

ndmm_net <- combine_network(
  set_ipd(ndmm_ipd,
          study, trt,
          Surv = Surv(eventtime / 12, status)),
  set_agd_surv(ndmm_agd,
               study, trt,
               Surv = Surv(eventtime / 12, status),
               covariates = ndmm_agd_covs))
# Plot KM curves using ggplot2
library(ggplot2)

# We need to create an empty ggplot object to add the curves to
ggplot() + geom_km(ndmm_net)

# Adding plotting options, facets, axis labels, and a plot theme
ggplot() +
  geom_km(ndmm_net,
          curve_args = list(linewidth = 0.5),
          cens_args = list(size = 3, shape = 124)) +
  facet_wrap(vars(Study)) +
  labs(xlab = "Time", ylab = "Survival Probability") +
  theme_multinma()

# Using the transform argument to produce log-log plots (e.g. to assess the
# proportional hazards assumption)
ggplot() +
  geom_km(ndmm_net, transform = "cloglog") +
  facet_wrap(vars(Study)) +
  theme_multinma()

# Using the transform argument to produce cumulative hazard plots
ggplot() +
  geom_km(ndmm_net, transform = "cumhaz") +
  facet_wrap(vars(Study)) +
  theme_multinma()

# This function can also be used to add KM data to plots of estimated survival
# curves from a fitted model, in a similar manner
\donttest{
# Run newly-diagnosed multiple myeloma example if not already available
if (!exists("ndmm_fit")) example("example_ndmm", run.donttest = TRUE)
}
# Plot estimated survival curves, and overlay the KM data
\donttest{
plot(predict(ndmm_fit, type = "survival")) + geom_km(ndmm_net)
}
}
