% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/netposet.R
\name{netposet}
\alias{netposet}
\alias{print.netposet}
\title{Partial order of treatments in network meta-analysis}
\usage{
netposet(
  ...,
  outcomes,
  treatments,
  small.values,
  common,
  random,
  fixed,
  comb.fixed,
  comb.random
)

\method{print}{netposet}(x, pooled = ifelse(x$random, "random", "common"), ...)
}
\arguments{
\item{\dots}{See details.}

\item{outcomes}{A character vector with outcome names.}

\item{treatments}{A character vector with treatment names.}

\item{small.values}{See details.}

\item{common}{A logical indicating whether to show results for the
common effects model.}

\item{random}{A logical indicating whether to show results for the
random effects model.}

\item{fixed}{Ignored deprecated argument (replaced by
\code{common}).}

\item{comb.fixed}{Ignored deprecated argument (replaced by
\code{common}).}

\item{comb.random}{Ignored deprecated argument (replaced by
\code{random}).}

\item{x}{An object of class \code{netposet}.}

\item{pooled}{A character string indicating whether Hasse diagram
should be drawn for common (\code{"common"}) or random effects
model (\code{"random"}). Can be abbreviated.}
}
\value{
An object of class \code{netposet} with corresponding \code{print},
\code{plot}, and \code{hasse} functions. The object is a list
containing the following components:
\item{P.common}{Ranking matrix with rows corresponding to treatments
  and columns corresponding to outcomes (common effects model).}
\item{M0.common}{Hasse matrix skipping unnecessary paths (common
  effects model).}
\item{M.common}{"Full" Hasse matrix (common effects model).}
\item{O.common}{Matrix with information about partial ordering
  (common effects model).}
\item{P.random}{Ranking matrix with rows corresponding to
  treatments and columns corresponding to outcomes (random effects
  model).}
\item{M0.random}{Hasse matrix skipping unnecessary paths (random
  effects model).}
\item{M.random}{"Full" Hasse matrix (random effects model).}
\item{O.random}{Matrix with information about partial ordering
  (random effects model).}
\item{small.values, common, random}{As.defined above.}
\item{call}{Function call.}
\item{version}{Version of R package netmeta used to create object.}
}
\description{
Partial order of treatments in network meta-analysis. The set of
treatments in a network is called a partially ordered set (in
short, a \emph{poset}), if different outcomes provide different
treatment ranking lists.
}
\details{
In network meta-analysis, frequently different outcomes are
considered which may each provide a different ordering of
treatments. The concept of a partially ordered set (in short, a
\emph{poset}, Carlsen & Bruggemann, 2014) of treatments can be used
to gain further insights in situations with apparently conflicting
orderings. This implementation for rankings in network meta-analyis
is described in Rücker & Schwarzer (2017).

In function \code{netposet}, argument \code{\dots{}} can be any of the
following:
\itemize{
\item arbitrary number of \code{netrank} objects providing
  P-scores;
\item arbitrary number of \code{netmeta} objects;
\item single ranking matrix with each column providing P-scores
  (Rücker & Schwarzer 2015) or SUCRA values (Salanti et al. 2011)
  for an outcome and rows corresponding to treatments.
}
Note, albeit in general a ranking matrix is not constrained to have
values between 0 and 1, \code{netposet} stops with an error in this
case as this function expects a matrix with P-scores or SUCRA
values.

Argument \code{outcomes} can be used to label outcomes. If argument
\code{outcomes} is missing,
\itemize{
\item column names of the ranking matrix are used as outcome labels
  (if first argument is a ranking matrix and column names are
  available);
\item capital letters 'A', 'B', \dots{} are used as outcome labels
  and a corresponding warning is printed.
}

Argument \code{treatments} can be used to provide treatment labels
if the first argument is a ranking matrix. If argument
\code{treatment} is missing,
\itemize{
\item row names of the ranking matrix are used as treatment labels
  (if available);
\item letters 'a', 'b', \dots{} are used as treatment labels and a
  corresponding warning is printed.
}
If argument \code{\dots{}} consists of \code{netmeta} objects,
\code{netrank} is called internally to calculate P-scores. In this
case, argument \code{small.values} can be used to specify for each
outcome whether small values are beneficial (\code{"desirable"}) or
harmfull (\code{"undesirable"}); see \code{\link{netrank}}. This
argument is ignored for a ranking matrix and \code{netrank}
objects.

Arguments \code{common} and \code{random} can be used to define
whether results should be printed and plotted for common and random
effects model. If netmeta and netrank objects are provided in
argument \code{\dots{}}, values for \code{common} and \code{random}
within these objects are considered; if these values are not
unique, argument \code{common} or \code{random} are set to
\code{TRUE}.

In function \code{print.netposet}, argument \code{\dots{}} is
passed on to the printing function.
}
\examples{
\donttest{
# Define order of treatments in depression data set dat.linde2015
#
trts <- c("TCA", "SSRI", "SNRI", "NRI",
  "Low-dose SARI", "NaSSa", "rMAO-A", "Hypericum", "Placebo")

# Outcome labels
#
outcomes <- c("Early response", "Early remission")

# (1) Early response
#
pw1 <- pairwise(treat = list(treatment1, treatment2, treatment3),
  event = list(resp1, resp2, resp3), n = list(n1, n2, n3),
  studlab = id, data = dat.linde2015, sm = "OR")
#
net1 <- netmeta(pw1, common = FALSE,
  seq = trts, ref = "Placebo", small.values = "undesirable")

# (2) Early remission
#
pw2 <- pairwise(treat = list(treatment1, treatment2, treatment3),
  event = list(remi1, remi2, remi3), n = list(n1, n2, n3),
  studlab = id, data = dat.linde2015, sm = "OR")
#
net2 <- netmeta(pw2, common = FALSE,
  seq = trts, ref = "Placebo", small.values = "undesirable")

# Partial order of treatment rankings (two outcomes)
#
po12 <- netposet(netrank(net1), netrank(net2), outcomes = outcomes)

# Scatter plot
#
plot(po12)

# Biplot (same scatter plot as only two outcomes considered)
#
plot(po12, "biplot")

# Hasse diagram for outcomes early response and early remission
#
\dontrun{
if (requireNamespace("Rgraphviz", quietly = TRUE))
  hasse(po12)
}

# Scatter plot for outcomes early response and early remission
#
oldpar <- par(pty = "s")
plot(po12)
par(oldpar)


# Consider five outcomes
#
# Outcome labels
#
outcomes <- c("Early response", "Early remission",
  "Lost to follow-up", "Lost to follow-up due to AEs",
   "Adverse events (AEs)")

# (3) Loss to follow-up
#
pw3 <- pairwise(treat = list(treatment1, treatment2, treatment3),
  event = list(loss1, loss2, loss3), n = list(n1, n2, n3),
  studlab = id, data = dat.linde2015, sm = "OR")
#
net3 <- netmeta(pw3, common = FALSE,
  seq = trts, ref = "Placebo", small.values = "desirable")

# (4) Loss to follow-up due to adverse events
#
pw4 <- pairwise(treat = list(treatment1, treatment2, treatment3),
  event = list(loss.ae1, loss.ae2, loss.ae3), n = list(n1, n2, n3),
  studlab = id, data = subset(dat.linde2015, id != 55), sm = "OR")
#
net4 <- netmeta(pw4, common = FALSE,
  seq = trts, ref = "Placebo", small.values = "desirable")

# (5) Adverse events
#
pw5 <- pairwise(treat = list(treatment1, treatment2, treatment3),
  event = list(ae1, ae2, ae3), n = list(n1, n2, n3),
  studlab = id, data = dat.linde2015, sm = "OR")
#
net5 <- netmeta(pw5, common = FALSE,
  seq = trts, ref = "Placebo", small.values = "desirable")

# Partial order of treatment rankings (based on netrank() objects)
#
po.ranks <- netposet(netrank(net1), netrank(net2),
  netrank(net3), netrank(net4), netrank(net5), outcomes = outcomes)

# Same result (based on netmeta() objects)
#
po.nets <- netposet(net1, net2, net3, net4, net5,
  outcomes = outcomes)
#
all.equal(po.ranks, po.nets)

# Print matrix with P-scores (random effects model)
#
po.nets$P.random

\dontrun{
# Hasse diagram for all outcomes (random effects model)
#
if (requireNamespace("Rgraphviz", quietly = TRUE))
  hasse(po.ranks)
}
}

# Example using ranking matrix with P-scores
#
# Ribassin-Majed L, Marguet S, Lee A.W., et al. (2017):
# What is the best treatment of locally advanced nasopharyngeal
# carcinoma? An individual patient data network meta-analysis.
# Journal of Clinical Oncology, 35, 498-505
#
outcomes <- c("OS", "PFS", "LC", "DC")
treatments <- c("RT", "IC-RT", "IC-CRT", "CRT",
  "CRT-AC", "RT-AC", "IC-RT-AC")
#
# P-scores (from Table 1)
#
pscore.os  <- c(15, 33, 63, 70, 96, 28, 45) / 100
pscore.pfs <- c( 4, 46, 79, 52, 94, 36, 39) / 100
pscore.lc  <- c( 9, 27, 47, 37, 82, 58, 90) / 100
pscore.dc  <- c(16, 76, 95, 48, 72, 32, 10) / 100
#
pscore.matrix <- data.frame(pscore.os, pscore.pfs, pscore.lc, pscore.dc)
rownames(pscore.matrix) <- treatments
colnames(pscore.matrix) <- outcomes
pscore.matrix
#
po <- netposet(pscore.matrix)
po12 <- netposet(pscore.matrix[, 1:2])
po
po12
#
\dontrun{
if (requireNamespace("Rgraphviz", quietly = TRUE)) {
  hasse(po)
  hasse(po12)
}
}
#
oldpar <- par(pty = "s")
plot(po12)
par(oldpar)

}
\references{
Carlsen L, Bruggemann R (2014):
Partial order methodology: a valuable tool in chemometrics.
\emph{Journal of Chemometrics},
\bold{28}, 226--34

Rücker G, Schwarzer G (2015):
Ranking treatments in frequentist network meta-analysis works
without resampling methods.
\emph{BMC Medical Research Methodology},
\bold{15}, 58

Rücker G, Schwarzer G (2017):
Resolve conflicting rankings of outcomes in network meta-analysis:
Partial ordering of treatments.
\emph{Research Synthesis Methods},
\bold{8}, 526--36

Salanti G, Ades AE, Ioannidis JP (2011):
Graphical methods and numerical summaries for presenting results
from multiple-treatment meta-analysis: an overview and tutorial.
\emph{Journal of Clinical Epidemiology},
\bold{64}, 163--71
}
\seealso{
\code{\link{netmeta}}, \code{\link{netrank}},
  \code{\link{plot.netrank}}, \code{\link{hasse}},
  \code{\link{plot.netposet}}, \code{\link[metadat]{dat.linde2015}}
}
\author{
Gerta Rücker \email{gerta.ruecker@uniklinik-freiburg.de}, Guido
  Schwarzer \email{guido.schwarzer@uniklinik-freiburg.de}
}
