rand.test.cdf2 <-
  function(x, y, 
           method = c("AD", "CVM", "KS"),
           R = 9999, parallel = FALSE, cl = NULL,
           perm.dist = TRUE){
    # Two-Sample Randomization Test for Distribution Equivalence
    # Nathaniel E. Helwig (helwig@umn.edu)
    # last updated: 2026-01-12
    
    
    #########   INITIAL CHECKS   #########
    
    ### check x and y
    x <- as.matrix(x)
    y <- as.matrix(y)
    m <- nrow(x)
    n <- nrow(y)
    nvar <- ncol(x)
    if(ncol(y) != nvar) stop("Inputs 'x' and 'y' must have the same number of columns.")
    N <- m + n
    
    ### check method
    methods <- c("AD", "CVM", "KS")
    method <- as.character(method[1])
    method <- pmatch(method, methods)
    if(is.na(method)) stop("Invalid 'method' input.")
    method <- methods[method]
    
    ### check R
    R <- as.integer(R)
    if(R < 1) stop("Input 'R' must be a positive integer.")
    
    ### check parallel
    parallel <- as.logical(parallel[1])
    
    ### check 'cl'
    make.cl <- FALSE
    if(parallel){
      if(is.null(cl)){
        make.cl <- TRUE
        cl <- parallel::makeCluster(2L)
      } else {
        if(!any(class(cl) == "cluster")) stop("Input 'cl' must be an object of class 'cluster'.")
      }
    }
    
    ### exact or approximate?
    suppressWarnings( nperm <- choose(N, n) )
    exact <- ifelse(nperm <= R + 1L, TRUE, FALSE)
    if(exact){
      ix <- combn(N, n)
      nperm <- ncol(ix)
    }
    
    
    #########   DISTRIBUTION TEST   #########
    
    ### univariate or multivariate
    if(nvar == 1L){
      
      ## UNIVARIATE TEST
      
      ## combine x and y
      z <- c(x, y)
      
      ## observed test statistic
      Tstat <- Tstat.cdf2(x = x, y = y, method = method)
      
      ## permutation distribution
      if(exact){
        
        # parallel or sequential computation?
        if(parallel){
          permdist <- parallel::parCapply(cl = cl, x = ix, 
                                          FUN = Tperm.cdf2, 
                                          z = z, ny = n,
                                          method = method,
                                          exact = exact)
        } else {
          permdist <- apply(X = ix, MARGIN = 2, 
                            FUN = Tperm.cdf2, 
                            z = z, ny = n,
                            method = method,
                            exact = exact)
        } # end if(parallel)
        
      } else {
        
        # approximate permutation test (given input R)
        nperm <- R + 1L
        permdist <- rep(0, nperm)
        permdist[1] <- Tstat
        
        # parallel or sequential computation?
        if(parallel){
          permdist[2:nperm] <- parallel::parSapply(cl = cl, X = integer(R), 
                                                   FUN = Tperm.cdf2, 
                                                   z = z, ny = n,
                                                   method = method,
                                                   exact = exact)
        } else {
          permdist[2:nperm] <- sapply(X = integer(R),
                                      FUN = Tperm.cdf2, 
                                      z = z, ny = n,
                                      method = method,
                                      exact = exact)
        } # end if(parallel)
        
      } # end if(exact)
      
      ## permutation p-value (greater than alternative = only option)
      p.value <- mean(permdist >= Tstat)
      
    } else {
      
      ## MULTIVARIATE TEST
      
      ## combine x and y
      z <- rbind(x, y)
      
      ## observed test statistic
      Tuni <- Tstat.cdf2.mv(x = x, y = y, method = method, combine = FALSE)
      Tstat <- max(Tuni)
      
      ## permutation distribution
      if(exact){
        
        # parallel or sequential computation?
        if(parallel){
          permdist <- parallel::parCapply(cl = cl, x = ix, 
                                          FUN = Tperm.cdf2.mv, 
                                          z = z, ny = n,
                                          method = method,
                                          exact = exact)
        } else {
          permdist <- apply(X = ix, MARGIN = 2, 
                            FUN = Tperm.cdf2.mv, 
                            z = z, ny = n,
                            method = method,
                            exact = exact)
        } # end if(parallel)
        
      } else {
        
        # approximate permutation test (given input R)
        nperm <- R + 1L
        permdist <- rep(0, nperm)
        permdist[1] <- Tstat
        
        # parallel or sequential computation?
        if(parallel){
          permdist[2:nperm] <- parallel::parSapply(cl = cl, X = integer(R), 
                                                   FUN = Tperm.cdf2.mv, 
                                                   z = z, ny = n,
                                                   method = method,
                                                   exact = exact)
        } else {
          permdist[2:nperm] <- sapply(X = integer(R),
                                      FUN = Tperm.cdf2.mv, 
                                      z = z, ny = n,
                                      method = method,
                                      exact = exact)
        } # end if(parallel)
        
      } # end if(exact)
      
      ## permutation p-value
      uni.p.value <- rep(NA, nvar)
      p.value <- mean(permdist >= Tstat)
      for(v in 1:nvar) uni.p.value[v] <- mean(permdist >= Tuni[v])
      
    } # end if(nvar == 1L)
    
    
    #########   RESULTS   #########
    
    ### return results
    if(make.cl) parallel::stopCluster(cl)
    if(!perm.dist) permdist <- NULL
    res <- list(statistic = Tstat, p.value = p.value,
                perm.dist = permdist, method = method,
                R = nperm - ifelse(exact, 0, 1), exact = exact)
    if(nvar > 1L) {
      res$univariate <- Tuni
      res$adj.p.values <- uni.p.value
    }
    class(res) <- "rand.test.cdf2"
    return(res)
    
  } # end rand.test.cdf2


### permutation replication (univariate)
Tperm.cdf2 <-
  function(i, z, ny, method = "AD", exact = FALSE){
    if(!exact) i <- sample.int(n = length(z), size = ny)
    Tstat <- Tstat.cdf2(x = z[-i], y = z[i], method = method)
    return(Tstat)
  } # end Tperm.cdf2.R

### permutation replication (multivariate)
Tperm.cdf2.mv <-
  function(i, z, ny, method = "AD", combine = TRUE, exact = FALSE){
    if(!exact) i <- sample.int(n = nrow(z), size = ny)
    Tstat <- Tstat.cdf2.mv(x = z[-i,], y = z[i,],
                           method = method, combine = combine)
    return(Tstat)
  } # end Tperm.cdf2.mv.R

### test statistic (univariate)
Tstat.cdf2 <- 
  function(x, y, method = "AD"){
    
    m <- length(x)
    n <- length(y)
    N <- m + n
    z <- c(x, y)
    
    if(method == "AD"){
      Fx <- ecdf(x)
      Fy <- ecdf(y)
      Fz <- ecdf(z)
      top <- (Fx(z) - Fy(z))^2
      bot <- Fz(z) * (1 - Fz(z))
      Tstat <- mean(top / bot, na.rm = TRUE)
    } else if(method == "CVM"){
      rz <- rank(z)
      rx <- rz[1:m]
      ry <- rz[(m+1):N]
      U <- (sum((rx - rank(x))^2) / n) + (sum((ry - rank(y))^2) / m)
      Tstat <- (U / (m * n)) - 2/3 + 1 / (6 * m * n)
    } else {
      Fx <- ecdf(x)
      Fy <- ecdf(y)
      Tstat <- max((Fx(z) - Fy(z))^2)
    } # end if(method == "AD")
    Tstat
  } # end Tstat.cdf2

### test statistic (univariate)
Tstat.cdf2.mv <- 
  function(x, y, method = "AD", combine = TRUE){
    nvar <- ncol(x)
    Tstat <- rep(0.0, nvar)
    for(j in 1:nvar) {
      Tstat[j] <- Tstat.cdf2(x = x[,j], y = y[,j], method = method)
    }
    if(combine) Tstat <- max(Tstat)
    Tstat
  } # end Tstat.cdf2.mv
