\name{np.cdf.test}
\alias{np.cdf.test}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{
Nonparametric Distribution Tests
}
\description{
Peforms one- or two-sample nonparametric (randomization) tests of cumulative distribution functions. Implements Anderson-Darling, Cramer-von Mises, and Kolmogorov-Smirnov test statistics.
}
\usage{
np.cdf.test(x, y = NULL,
            method = c("AD", "CVM", "KS"),
            R = 9999, parallel = FALSE, cl = NULL,
            perm.dist = TRUE, na.rm = TRUE)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{
Numeric vector (or matrix) of data values.
}
  \item{y}{
One-sample: name of distribution family with "p" and "r" components (see Note). Two-sample: numeric vector (or matrix) of data values.
}
  \item{method}{
Test statistic to use: AD = Anderson-Darling, CVM = Cramer-Von Mises, KS = Kolmogorov-Smirnov
}
  \item{R}{
Number of resamples for the permutation test (positive integer). 
}
  \item{parallel}{
Logical indicating if the \code{\link{parallel}} package should be used for parallel computing (of the permutation distribution). Defaults to \code{FALSE}, which implements sequential computing.
}
  \item{cl}{
Cluster for parallel computing, which is used when \code{parallel = TRUE}. Note that if \code{parallel = TRUE} and \code{cl = NULL}, then the cluster is defined as \code{makeCluster(2L)} to use two cores. To make use of all available cores, use the code \code{cl = makeCluster(detectCores())}.
}
  \item{perm.dist}{
Logical indicating if the permutation distribution should be returned.  
}
  \item{na.rm}{
If \code{TRUE} (default), the arguments \code{x} and \code{groups} (and \code{blocks} if provided) are passed to the \code{\link{na.omit}} function to remove cases with missing data.
}
}
\details{
One-sample statistics:
\tabular{lll}{
\code{AD} \tab \tab \eqn{\omega^2 = \int w(x) (F_n(x) - F_0(x))^2 d F_0(x)} with \eqn{w(x) = [F_0(x)(1 - F_0(x))]^{-1}} \cr
\code{CVM} \tab \tab \eqn{\omega^2 = \int w(x) (F_n(x) - F_0(x))^2 d F_0(x)} with \eqn{w(x) = 1} \cr
\code{KS} \tab \tab \eqn{\omega^2 = \sup_{x} (F_n(x) - F_0(x))^2}  \cr
}
where \eqn{F_n(x)} is the empirical cumulative distribution function (estimated by \code{\link{ecdf}}) and \eqn{F_0} is the null hypothesized distribution (specified by the \code{y} argument).

Two-sample statistics:
\tabular{lll}{
\code{AD} \tab \tab \eqn{\omega^2 = \int w(z) (F_{x}(z) - F_{y}(z))^2 d F_0(z)} with \eqn{w(z) = [F_0(z)(1 - F_0(z))]^{-1}} \cr
\code{CVM} \tab \tab \eqn{\omega^2 = \int w(z) (F_{x}(z) - F_{y}(z))^2 d F_0(z)} with \eqn{w(z) = 1} \cr
\code{KS} \tab \tab \eqn{\omega^2 = \sup_{z} (F_{x}(z) - F_{y}(z))^2}  \cr
}
where \eqn{F_x} and \eqn{F_y} are the groupwise ECDF functions (estimated by applying \code{\link{ecdf}} separately to \code{x} and \code{y}) and \eqn{F_0} is the joint ECDF (estimated by applying \code{\link{ecdf}} to \code{z = c(x,y)} ). 

}
\value{
\item{statistic }{
Test statistic value.
}
  \item{p.value }{
p-value for testing \eqn{H_0: F_x = F_0} or \eqn{H_0: F_x = F_y}.
}
  \item{perm.dist }{
Permutation distribution of \code{statistic}. 
}
  \item{method }{
Method used for permutation test. See Examples.  
} 
  \item{R }{
Number of resamples.
}
  \item{exact }{
Exact permutation test? See Note.
}
}
\references{
Anderson, T. W. and Darling., D. A. (1952). Asymptotic theory of certain "goodness of fit" criteria based on stochastic processes. \emph{Annals of Mathematical Statistics, 23}(2), 193-212. \doi{10.1214/aoms/1177729437}

Anderson, T. W., and Darling, D. A. (1954). A test of goodness of fit. \emph{Journal of the American Statistical Association, 49}(268), 765-769. \doi{10.1080/01621459.1954.10501232}

Anderson, T. W. (1962). On the distribution of the two-sample Cramer-von Mises criterion. \emph{Annals of Mathematical Statistics, 33}(3) 1148-1159. \doi{10.1214/aoms/1177704477}

Cramer, H. (1928). On the composition of elementary errors: First paper: Mathematical deductions. \emph{Scandinavian Actuarial Journal, 1928}(1), 13-74. \doi{10.1080/03461238.1928.10416862}

Kolmogorov, A. N. (1933). Sulla determinazione empirica di una legge di distribuzione. Giornale dell'Istituto Italiano degli Attuari 4, 83-91.

Kolmogorov, A. N. (1941). Confidence limits for an unknown distribution function. \emph{Annals of Mathematical Statistics 12}(4), 461-483. \doi{10.1214/aoms/1177731684}

Smirnov, N. (1948). Table for estimating the goodness of fit of empirical distributions. \emph{Annals of Mathematical Statistics 19}(2) 279-281. \doi{10.1214/aoms/1177730256}

von Mises, R. (1928). Wahrscheinlichkeit, Statistik und Wahrheit. Julius Springer. 
}
\author{
Nathaniel E. Helwig <helwig@umn.edu>
}
\note{
For one-sample tests, the \code{y} argument should satisfy: \cr 
\code{paste("p", y)} gives the name of a function specifying the CDF \cr
\code{paste("r", y)} gives the name of a function sampling from the distribution \cr

If \code{y = NULL}, the default sets \code{y = "norm"}, which tests the null hypothesis that \code{x} follows a standard normal distribution. See the examples for how to test a user-specified distribution.
}

%% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{
\code{\link{plot.np.cdf.test}} S3 plotting method for visualizing the results
}
\examples{

###***###   ONE SAMPLE   ###***###

## generate standard normal data
n <- 100
set.seed(0)
x <- rnorm(n)


## Example 1: Fn = norm,  F0 = norm

# Anderson-Darling test of H0: Fx = pnorm
set.seed(1)
np.cdf.test(x, y = "norm")

\dontrun{

# Cramer-von Mises test of H0: Fx = pnorm
set.seed(1)
np.cdf.test(x, y = "norm", method = "CVM")

# Kolmogorov-Smirnov test of H0: Fx = pnorm
set.seed(1)
np.cdf.test(x, y = "norm", method = "KS")


## Example 2: Fn = norm,  F0 = t3

# user-defined distribution (Student's t with df = 3)
pt3 <- function(q) pt(q, df = 3)      # cdf = paste("p", y)
rt3 <- function(n) rt(n, df = 3)      # sim = paste("r", y)

# Anderson-Darling test of H0: Fx = t3
set.seed(1)
np.cdf.test(x, y = "t3")

# Cramer-von Mises test of H0: Fx = t3
set.seed(1)
np.cdf.test(x, y = "t3", method = "CVM")

# Kolmogorov-Smirnov test of H0: Fx = t3
set.seed(1)
np.cdf.test(x, y = "t3", method = "KS")



###***###   TWO SAMPLE   ###***###

# generate N(0, 1) and N(2/3, 1) data
m <- 25
n <- 25
set.seed(0)
x <- rnorm(m)
y <- rnorm(n, mean = 2/3)

# Anderson-Darling test of H0: Fx = Fy
set.seed(1)
np.cdf.test(x, y)

# Cramer-von Mises test of H0: Fx = Fy
set.seed(1)
np.cdf.test(x, y, method = "CVM")

# Kolmogorov-Smirnov test of H0: Fx = Fy
set.seed(1)
np.cdf.test(x, y, method = "KS")
}
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory (show via RShowDoc("KEYWORDS")):
\keyword{ htest }
\keyword{ nonparametric }
\keyword{ multivariate }
\keyword{ robust }
\keyword{ univar }
