#' Computes Bowker's test of symmetry.
#'
#' Computes the test of table symmetry in Bowker (1948).
#' Bowker, A. H. (1948). A test for symmetry in contingency tables.
#' Journal of the American Statistical Association 43, 572-574.
#' @importFrom stats pchisq
#' @param n the matrix to be tested for symmetry
#' @returns a list containing the
#'    chi-square: Pearson X^2
#'    g_square: likelihood ratio G^2
#'    df: degrees of freedom
#'    p-value: p-value for Pearson X^2
#'    expected: fitted values
#' @export
#' @examples
#' Bowker_symmetry(vision_data)
Bowker_symmetry <- function(n) {
  r <- nrow(n)
  k <- r - 1
  N <- sum(n)
  expected <- matrix(nrow=nrow(n), ncol=ncol(n))

  chisq <- 0.0
  df <- 0
  for (i in 1:k) {
    expected[i, i] <- n[i, i]
    for (j in (i + 1): r) {
      expected[i, j] <- (n[i, j] + n[j, i]) / 2.0
      expected[j, i] <- expected[i, j]
      if (n[i, j] > 0 || n[j, i] > 0) {
        df <- df + 1
        chisq <- chisq + (n[i, j] - n[j, i])^2 / (n[i, j] + n[j, i])
      }
    }
  }
  expected[r, r] <- n[r, r]
  p <- pchisq(chisq, df, lower.tail = FALSE)
  g_squared <- likelihood_ratio_chisq(n, expected / N)
  list(chisq=chisq, g_squared=g_squared, df=df, p=p, expected=expected)
}
