# McCullagh's quasu-symmetry model.

#' Fits McCullagh's (1978) quasi-symmetry model.
#'
#' McCullagh, P. (1978).  A class of parametric models for the analysis of
#' square contingency tables with ordered categories.  Biometrika, 65(2) 413-418.
#' @param n matrix of observed counts
#' @param max_iter maximum number of iterations in maximizing log(likelihood), Default is 15.
#' @param verbose should cycle-by-cycle information be printed out? Default is FALSE,
#' do not print
#' @returns a list containing
#'    phi: symmetry matrix
#'    alpha: vector of asymmetry parameters
#'    chisq: Pearson chi-square value
#'    df; degrees of freedom
#' @export
#' @examples
#' McCullagh_quasi_symmetry(vision_data)
McCullagh_quasi_symmetry <- function(n, max_iter=15, verbose=FALSE) {
  r <- nrow(n)
  phi <- McCullagh_q_symmetry_initialize_phi(r)
  alpha <- McCullagh_q_symmetry_initialize_alpha(r)

  if (verbose) {
    message("from q-symmetry\n")
    message(paste("0: ", log_likelihood(n, pi), "\n"))
  }

  for (iter in 1:max_iter) {
    result <- McCullagh_maximize_q_symmetry(n, phi, alpha)
    phi <- result$phi
    alpha <- result$alpha
    pi <- McCullagh_q_symmetry_pi(phi, alpha)
    if (verbose) {
      message(paste(iter, ": ", log_likelihood(n, pi), "\n"))
    }
  }
  pi <- McCullagh_q_symmetry_pi(phi, alpha)
  chisq = pearson_chisq(n, pi)
  df <- (r - 1) * (r - 2) / 2

  list(phi=phi, alpha=alpha, chisq=chisq, df=df)
}


#' Initializes the phi matrix
#'
#' @param M size of the psi matrix to create
#' @returns the symmetry matrix phi
McCullagh_q_symmetry_initialize_phi <- function(M) {
  phi <- matrix(0.2, nrow=M, ncol=M)
  for (i in 1:M) {
    phi[i, i] <- 1.0
  }
  const <- sum(phi)
  phi <- phi / const
}


#' Initializes the asymmetry vector alpha
#'
#' @param M size of alpha vector to create = nrow(matrix to analyze)
#' @returns vector of asymmetry parameters alpha
McCullagh_q_symmetry_initialize_alpha <- function(M) {
  alpha <- rep(1.0, M)
}


#' Computes the model-based p-values
#'
#' @param phi the matrix of symmetry parameters
#' @param alpha the vector of asymmetry parameters
#' @returns matrix pi of model-based p-values
McCullagh_q_symmetry_pi <- function(phi, alpha) {
  M = nrow(phi)
  pi <- matrix(0.0, nrow=nrow(phi), ncol=ncol(phi))

  for (i in 1:M) {
    for (j in 1:M) {
      pi[i, j] <- phi[i, j] * alpha[i] / alpha[j]
    }
    pi[i, i] <- phi[i, i]
  }

  const <- sum(pi)
  pi <- pi / const
}


#' Compute sums too use in maximizing log(likelihood)
#'
#' @param n matrix of observed counts
#' @returns list of s_i_plus and s_plus_i
McCullagh_compute_s_plus <- function(n) {
  M = nrow(n)
  s_i_plus <-rep(0.0, M)
  s_plus_i <- rep(0.0, M)

  for (i in 1:M) {
    for (j in 1:M) {
      if (j == i) {
        next
      }
      s_i_plus[i] <- s_i_plus[i] + n[i, j]
      s_plus_i[i] <- s_plus_i[i] + n[j, i]
    }
  }

  list(s_i_plus=s_i_plus, s_plus_i=s_plus_i)
}


#' Computes sums c+ used in maximizing the log(likelihod)
#'
#' @param phi matrix of symmetry parameters
#' @param alpha vector of asymmetry parameters
#' @returns list of c_i_plus and c_plus_i
McCullagh_compute_c_plus <- function(phi, alpha) {
  M <- nrow(phi)
  c_i_plus <- rep(0.0, M)
  c_plus_i <- rep(0.0, M)

  c_i_plus[1] <- phi[1, 1] / alpha[1]
  c_plus_i[1] <- phi[1, 1] * alpha[1]

  for (i in 1:M) {
    for (j in 1:M) {
      if (j == i) {
        next
      }
      c_i_plus[i] <- c_i_plus[i] + phi[i, j] / alpha[j]
      c_plus_i[i] <- c_plus_i[i] + phi[i, j] * alpha[j]
    }
  }
  list(c_i_plus=c_i_plus, c_plus_i=c_plus_i)
}


#' Maximize the log(likelihood) wrt parameters phi and alpha
#'
#' @param n matrix of observed counts
#' @param phi matrix of symmetry parameters
#' @param alpha vector of asymmetry parameters
#' @returns list with new values of phi and alpha
#' @export
McCullagh_maximize_q_symmetry <- function(n, phi, alpha) {
  r <- nrow(n)
  N <- sum(n)

  c_plus <- McCullagh_compute_c_plus(phi, alpha)
  c_i_plus <- c_plus$c_i_plus
  c_plus_i <- c_plus$c_plus_i

  s_plus <- McCullagh_compute_s_plus(n)
  s_i_plus <- s_plus$s_i_plus
  s_plus_i <- s_plus$s_plus_i

  for (i in 2:r) {
    K <- s_i_plus[i] - s_plus_i[i]

    a <- c_i_plus[i]
    b <- -K / N
    c <- - c_plus_i[i]
    discrim <- b^2 - 4.0 * a * c
    if (discrim < 0.0) {
      discrim <- 0.0
    }
    root1 <- (-b + sqrt(discrim)) / (2.0 * a)
    root2 <- (-b - sqrt(discrim)) / (2.0 * a)
    alpha[i] <- root1
  }

  phi[1, 1] <- n[1, 1] / N
  for (i in 2: r) {
    for (j in 1:i - 1) {
      phi[i, j] <- (n[i, j] + n[j, i]) / (N * (alpha[i] / alpha[j]
                                        + alpha[j] / alpha[i]))
      phi[j, i] <- phi[i, j]
    }
    phi[i, i] <- n[i, i] / N
  }

  list(phi=phi, alpha=alpha)
}
